package de.narimo.commons;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import com.vividsolutions.jts.geom.Envelope;
import com.vividsolutions.jts.geom.Geometry;
import com.vividsolutions.jts.io.ParseException;
import com.vividsolutions.jts.io.WKTReader;

import de.narimo.commons.dto.BBox;
import de.narimo.commons.dto.xml.ows.BoundingBox;

public class BBoxTools {

    public static enum AxisOrder {
        X_Y,
        Y_X;
    }

    /**
     * Converts a delimiter-separated string to a bbox. Respects axis order. Default
     * delimiter is comma.
     * 
     * @param gnGeoBox
     * @return
     * @throws Exception
     */
    public static BBox toBBox(String gnGeoBox, String optDelimiter, AxisOrder axisOrder) throws Exception {

        // "geoBox": "-8.6500|49.7700|1.7800|60.8600",

        if (gnGeoBox == null) {
            return null;
        }

        if (optDelimiter == null) {
            optDelimiter = ",";
        }
        String[] split = gnGeoBox.split("\\" + optDelimiter);

        if (split.length != 4) {
            throw new IllegalArgumentException("BBox must consist of four coordinates.");
        }

        BBox bbox = new BBox();
        if (axisOrder.equals(AxisOrder.X_Y)) {
            bbox.setXMin(Double.valueOf(split[0]));
            bbox.setYMin(Double.valueOf(split[1]));
            bbox.setXMax(Double.valueOf(split[2]));
            bbox.setYMax(Double.valueOf(split[3]));
        } else if (axisOrder.equals(AxisOrder.Y_X)) {
            bbox.setYMin(Double.valueOf(split[0]));
            bbox.setXMin(Double.valueOf(split[1]));
            bbox.setYMax(Double.valueOf(split[2]));
            bbox.setXMax(Double.valueOf(split[3]));
        } else {
            throw new Exception("Unknown axis order " + axisOrder);
        }

        return bbox;
    }

    public static BBox toBBox(BoundingBox boundingBox, AxisOrder axisOrder) throws Exception {
        if (boundingBox == null) {
            return null;
        }
        List<Double> lowerCorner = cornerToCoordinateTuple(boundingBox.getLowerCorner());
        List<Double> upperCorner = cornerToCoordinateTuple(boundingBox.getUpperCorner());

        // expecting lat,lon order by default
        BBox bbox = new BBox();
        if (axisOrder == null || axisOrder.equals(AxisOrder.Y_X)) {
            bbox.setXMin(lowerCorner.get(1));
            bbox.setYMin(lowerCorner.get(0));
            bbox.setXMax(upperCorner.get(1));
            bbox.setYMax(upperCorner.get(0));
        } else if (axisOrder.equals(AxisOrder.X_Y)) {
            bbox.setYMin(lowerCorner.get(1));
            bbox.setXMin(lowerCorner.get(0));
            bbox.setYMax(upperCorner.get(1));
            bbox.setXMax(upperCorner.get(0));
        } else {
            throw new Exception("Unknown axis order " + axisOrder);
        }
        return bbox;
    }

    private static List<Double> cornerToCoordinateTuple(String cornerCoordinates) {
        List<String> cornerArray = Arrays.asList(cornerCoordinates.split("\\s"));
        List<Double> corner = new ArrayList<Double>();
        for (String c : cornerArray) {
            try {
                corner.add(Double.parseDouble(c));
            } catch (NumberFormatException e) {
                continue;
            }
        }
        return corner;
    }

    public static BBox[] toBBoxArray(List<String> gnGeoBoxArray, String optDelimiter, AxisOrder axisOrder)
            throws Exception {

        if (gnGeoBoxArray == null) {
            return null;
        }

        BBox[] bboxArray = new BBox[gnGeoBoxArray.size()];

        int i = 0;
        for (String b : gnGeoBoxArray) {

            bboxArray[i] = toBBox(b, optDelimiter, axisOrder);
            i++;
        }
        return bboxArray;
    }

    public static String bboxArrayToWKTMultipolygon(BBox[] bboxArray) {
        if (bboxArray == null) {
            return null;
        }

        String bboxWKT = "Multipolygon(";
        int i = 0;
        for (BBox bbox : bboxArray) {
            if (bbox == null) {
                continue;
            }
            if (i > 0) {
                bboxWKT += ",";
            }
            bboxWKT += "((" +
                    bbox.getXMin() + " " + bbox.getYMax() + "," +
                    bbox.getXMax() + " " + bbox.getYMax() + "," +
                    bbox.getXMax() + " " + bbox.getYMin() + "," +
                    bbox.getXMin() + " " + bbox.getYMin() + "," +
                    bbox.getXMin() + " " + bbox.getYMax() + "))";

            i++;
        }
        bboxWKT += ")";
        return bboxWKT;
    }

    public static String bboxToWktPolygon(BBox bbox) {
        if (bbox == null) {
            bbox = new BBox();
            bbox.XMax = 180;
            bbox.XMin = -180;
            bbox.YMax = 90;
            bbox.YMin = -90;
        }

        String bboxWKT = "Multipolygon(";
        bboxWKT += "((" +
                bbox.getXMin() + " " + bbox.getYMax() + "," +
                bbox.getXMax() + " " + bbox.getYMax() + "," +
                bbox.getXMax() + " " + bbox.getYMin() + "," +
                bbox.getXMin() + " " + bbox.getYMin() + "," +
                bbox.getXMin() + " " + bbox.getYMax() + "))";
        bboxWKT += ")";
        return bboxWKT;
    }

    /**
     * Creates a BBOX from a WKT String.
     * 
     * @param wkt
     * @return
     */
    public static BBox wktToBBox(String wkt) {
        if (wkt == null) {
            return null;
        }

        WKTReader reader = new WKTReader();
        Geometry g;

        try {
            g = reader.read(wkt);
        } catch (ParseException e) {
            return null;
        }

        Envelope env = g.getEnvelopeInternal();

        BBox bbox = new BBox();
        bbox.setXMin(env.getMinX());
        bbox.setYMin(env.getMinY());
        bbox.setXMax(env.getMaxX());
        bbox.setYMax(env.getMaxY());
        return bbox;
    }

    /**
     * Returns a comma-separated string of the bbox. First tuple is LL, second tuple
     * is UR corner. Respects axis ordering.
     *
     * @param axisOrder
     */
    public static String toString(BBox bbox, AxisOrder axisOrder) {

        if (axisOrder.equals(AxisOrder.X_Y)) {
            return bbox.getXMin() + "," + bbox.getYMin() + "," + bbox.getXMax() + "," + bbox.getYMax();
        } else if (axisOrder.equals(AxisOrder.Y_X)) {
            return bbox.getYMin() + "," + bbox.getXMin() + "," + bbox.getYMax() + "," + bbox.getXMax();
        }
        return null;
    }

    public static double constrainToWGS84Box(double coordinate, String axis) {
        if (axis.equals("x")) {

            if (coordinate < -180) {
                return -180;
            }
            if (coordinate > 180) {
                return 180;
            }
        }
        if (axis.equals("y")) {

            if (coordinate < -90) {
                return -90;
            }
            if (coordinate > 90) {
                return 90;
            }
        }
        return coordinate;
    }

    public static void main(String[] args) {

        //
        // double xmax = 10.5;
        // double xmin = -3.2;
        // double ymax = 53.4;
        // double ymin = 49.9;
        ////
        //// int size = 150;
        ////
        //// double widthDeg = Math.abs(xmax-xmin);
        ////// double ratio = size/widthDeg;
        ////
        //// double heightDeg = Math.abs(ymax-ymin);
        //// double ratio = size/heightDeg;
        ////
        ////
        //// double widthPx = widthDeg*ratio;
        //// double heightPx = size;
        ////
        //// System.out.println("Ratio: "+ratio);
        //// System.out.println("widthDeg: "+widthDeg);
        //// System.out.println("heightDeg: "+heightDeg);
        ////
        //// System.out.println("Tile width: "+widthPx);
        //// System.out.println("Tile height: "+heightPx);
        ////
        ////
        //// if(widthPx>size){
        ////
        //// widthPx = size;
        //// heightPx = heightDeg*size/widthDeg;
        //// }
        //
        // BBox bbox = new BBox();
        // bbox.setXMin(xmin);
        // bbox.setYMin(ymin);
        // bbox.setXMax(xmax);
        // bbox.setYMax(ymax);
        //
        // String ts = getTileSize(bbox, 150);
        //
        // String[] tsparts = ts.split("x");
        //
        // String widthPx = tsparts[0];
        // String heightPx = tsparts[1];
        //
        // System.out.println("Tile width: "+widthPx);
        // System.out.println("Tile height: "+heightPx);
        //
        //// WKTReader reader = new WKTReader();
        //// String wkt = "MULTIPOLYGON (((40 40, 20 45, 45 30, 40 40)), ((20 35, 55 55,
        // 10 10, 30 5, 45 20, 20 35),(30 20, 20 15, 20 25, 30 20)), 4326)";
        ////
        //// Geometry g = null;
        //// try {
        //// g = reader.read(wkt);
        //// } catch (ParseException e) {
        //// e.printStackTrace();
        ////// return null;
        //// }
        ////
        ////// EPSG:4326
        ////// g.getSRID();
        ////
        //// System.out.println(g.getSRID());
        //
        //
        //
        ////// String wkt = "POINT (1.0 2.0)";
        ////// Point point = (Point) reader.read(wkt);
        ////
        ////
        //// String wkt = "MULTIPOLYGON (((40 40, 20 45, 45 30, 40 40)), ((20 35, 55 55,
        // 10 10, 30 5, 45 20, 20 35),(30 20, 20 15, 20 25, 30 20)))";
        //// WKTReader reader = new WKTReader();
        ////
        //// MultiPolygon mp = (MultiPolygon) reader.read(wkt);
        ////
        //// System.out.println(mp.getNumberOfCoordinates());
        ////
        //// System.out.println(mp.getDimension());
        ////
        //// List<Double> numsX = new ArrayList<Double>();
        //// List<Double> numsY = new ArrayList<Double>();
        ////
        //// for(Polygon p : mp.getPolygons()){
        ////
        //// LinearRing or = p.getOuterLinearRing();
        ////
        //// for (Coordinate c : or.getCoordinates()){
        ////// Coordinate c = or.getCoordinates().get(index);
        //// numsX.add(c.getX());
        //// numsY.add(c.getY());
        //// }
        //// }
        ////
        ////
        //// Collections.sort(numsX);
        //// Collections.sort(numsY);
        ////
        ////// Double min = numsX.get(0);
        ////// Double max = numsX.get(numsX.size()-1);
        ////
        ////
        ////// System.out.println(numsX.toString());
        //////
        ////// System.out.println("minX "+min);
        ////// System.out.println("maxX "+max);
        //////
        //////
        ////// System.out.println("minY "+numsY.get(0));
        ////// System.out.println("maxY "+numsY.get(numsY.size()-1));
        ////
        //// BBox bbox = new BBox();
        //// bbox.setXMin(numsX.get(0));
        //// bbox.setYMin(numsY.get(0));
        //// bbox.setXMax(numsX.get(numsX.size()-1));
        //// bbox.setYMax(numsY.get(numsY.size()-1));
        //
    }
}
