package de.narimo.commons.jdbc;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;

import javax.naming.InitialContext;
import javax.sql.DataSource;

public class JDBCConnectionJNDI {

    private Connection connection;
    private List<Statement> statements = new ArrayList<Statement>();
    private List<ResultSet> resultSets = new ArrayList<ResultSet>();
    private List<PreparedStatement> preparedStatements = new ArrayList<PreparedStatement>();

    private int connectionId;

    private boolean autoCommitDefault;

    /**
     * JNDI Resource name as defined in <resource-ref> in web.xml.
     *
     * @param JNDIResource
     * @throws Exception
     */
    public JDBCConnectionJNDI(String JNDIResource) throws Exception {

        DataSource ds = (DataSource) new InitialContext().lookup("java:/comp/env/" + JNDIResource);
        if (ds == null) {
            throw new Exception("Data source not found!");
        }

        this.connection = ds.getConnection();

        this.connectionId = (int) (Math.random() * 1000.0);
        System.out.println("Connection with id " + this.connectionId + " was opened.");

        this.autoCommitDefault = this.connection.getAutoCommit();
        // this.autoCommitDefault = true; //should be set by pool itself

        // this.connection.setAutoCommit(true); //set explicitly to true,
        // although is the default
    }

    public ResultSet executeQuery(String sql) throws Exception {

        Statement stmt = this.connection.createStatement();
        this.statements.add(stmt);

        ResultSet rs = stmt.executeQuery(sql);
        this.resultSets.add(rs);

        return rs;

    }

    public boolean execute(String sql) throws Exception {

        Statement stmt = this.connection.createStatement();
        this.statements.add(stmt);

        return stmt.execute(sql);

    }

    /**
     * Returns a PreparedStatement. The PreparedStatement must be explicitly
     * closed after use, to prevent connection leaks!
     *
     * @param sql
     * @return
     * @throws Exception
     */
    public PreparedStatement prepareStatement(String sql) throws Exception {

        PreparedStatement ps = this.connection.prepareStatement(sql);
        this.preparedStatements.add(ps);

        return ps;
    }

    /**
     * Returns a ResultSet. The ResultSet must be explicitly closed after use,
     * to prevent connection leaks!
     *
     * @param ps
     * @return
     * @throws Exception
     */
    public ResultSet executePreparedQuery(PreparedStatement ps) throws Exception {

        ResultSet rs = ps.executeQuery();
        this.resultSets.add(rs);

        return rs;
    }

    public int executePreparedUpdate(PreparedStatement ps) throws Exception {

        int res = ps.executeUpdate();

        return res;
    }

    /**
     * @deprecated
     * @throws SQLException
     */
    @Deprecated
    public void close() throws SQLException {
        this.closeAll();
        // if(this.connection!=null) {
        // this.connection.close();
        // System.out.println("Connection with id "+this.connectionId+" was
        // closed.");
        // }
    }

    public void closeAll() {

        try {
            for (ResultSet rs : this.resultSets) {
                if (rs != null) {
                    rs.close();
                }
                rs = null;
            }

            for (Statement stmt : this.statements) {
                if (stmt != null) {
                    stmt.close();
                }
                stmt = null;
            }

            for (PreparedStatement ps : this.preparedStatements) {
                if (ps != null) {
                    ps.close();
                }
                ps = null;
            }

            if (this.connection != null) {
                this.connection.setAutoCommit(this.autoCommitDefault);
                this.connection.close();
                this.connection = null;
                System.out.println("Connection with id " + this.connectionId + " was closed.");
            }
        } catch (SQLException e) {
            e.printStackTrace();
        }
    }

    // public void closeAll(Statement stmt, ResultSet rs) throws SQLException{
    //
    // if(rs!=null) rs.close();
    // rs=null;
    //
    // if(stmt!=null) stmt.close();
    // stmt=null;
    //
    // if(this.connection!=null) {
    // this.connection.close();
    // this.connection = null;
    // System.out.println("Connection with id "+this.connectionId+" was
    // closed.");
    // }
    // }

    public void commit() throws SQLException {
        this.connection.commit();
    }

    public void rollback() throws SQLException {
        this.connection.rollback();
    }

    public void setAutoCommit(boolean autoCommit) throws SQLException {
        this.connection.setAutoCommit(autoCommit);
    }

    public int getConnectionId() {
        return this.connectionId;
    }

    public boolean isClosed() throws SQLException {
        return this.connection.isClosed();
    }

    public boolean getAutoCommit() throws SQLException {
        return this.connection.getAutoCommit();
    }

    public static void main(String[] args) throws SQLException {

    }
}
