package de.narimo.geocore.ws.auth;

import java.io.UnsupportedEncodingException;
import java.util.Calendar;
import java.util.Date;

import com.auth0.jwt.JWT;
import com.auth0.jwt.JWTCreator.Builder;
import com.auth0.jwt.JWTVerifier;
import com.auth0.jwt.algorithms.Algorithm;
import com.auth0.jwt.exceptions.JWTCreationException;
import com.auth0.jwt.exceptions.JWTDecodeException;
import com.auth0.jwt.interfaces.Claim;
import com.auth0.jwt.interfaces.DecodedJWT;

public class JWTFactory {

	private String secret;
	private String issuer;

	//custom field holding an access token that must be presented to the underlying authorization component
	private String atn;
	private static String ACCESS_TOKEN_IDENTIFIER = "atn";

	public JWTFactory(String secret, String issuer){
		this.secret = secret;
		this.issuer = issuer;
	}

	public void setAccessToken(String authorizationToken){
		this.atn = authorizationToken;
	}

	public String createToken(int validityMins){

		String jwtToken = null;
		Date expirationDate = null;

		if(validityMins>0){
			Calendar cal = Calendar.getInstance();
			cal.setTimeInMillis(System.currentTimeMillis()+(validityMins*60*1000));
			expirationDate = cal.getTime();
		}

		try {

			Algorithm algorithm = Algorithm.HMAC256(this.secret);

			Builder b = JWT.create()
					.withIssuer(this.issuer);

			if(expirationDate!=null) {
				b.withExpiresAt(expirationDate);
			}

			if(this.atn!=null) {
				b.withClaim(ACCESS_TOKEN_IDENTIFIER, this.atn);
			}

			jwtToken = b.sign(algorithm);
		} catch (UnsupportedEncodingException exception){
			//UTF-8 encoding not supported
			exception.printStackTrace();
		} catch (JWTCreationException exception){
			//Invalid Signing configuration / Couldn't convert Claims.
			exception.printStackTrace();
		}

		if(jwtToken==null) {
			throw new IllegalArgumentException("No token was created due to incorrect claims.");
		}
		return jwtToken;
	}

	/**
	 * Verifies a provided jwt token and returns true on success.
	 * Throws an exception in all other cases.
	 *
	 * @param jwtToken
	 * @return
	 * @throws Exception
	 */
	public boolean verifyToken(String jwtToken) throws Exception{

		if(jwtToken==null) {
			return false;
		}

		Algorithm algorithm = Algorithm.HMAC256(this.secret);
		JWTVerifier verifier = JWT.require(algorithm)
				.withIssuer(this.issuer)
				.build(); //Reusable verifier instance
		verifier.verify(jwtToken);

		return true;
	}

	/**
	 * Returns the saved access token ("atn") which can be used for authorization at an underlying security system.
	 *
	 * @return
	 */
	public String getAccessToken(String jwtToken){
		try {
			DecodedJWT jwt = JWT.decode(jwtToken);

			Claim accessTokenClaim = jwt.getClaim("atn");
			if(accessTokenClaim.isNull()) {
				return null;
			}
			return accessTokenClaim.asString();

		} catch (JWTDecodeException exception){
			//Invalid token
		}
		return null;
	}

	public static void main(String args[]){

	}
}



