package de.narimo.geocore.ws.auth.filter;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;

import javax.servlet.FilterConfig;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.NotFoundException;
import javax.ws.rs.container.ContainerRequestContext;
import javax.ws.rs.container.ContainerRequestFilter;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;

import de.narimo.commons.UserCredentials;
import de.narimo.geocore.ws.auth.OverrideSecurityContext;
import de.narimo.geocore.ws.auth.tools.AuthenticationFilterTools;

/**
 *
 * Request filter class for basic authentication. This filter class should not
 * be used for authentication itself. Use Filter implementations overriding
 * authenticate() method. If no authentication details have been provided,
 * filter refuses a request including www-authenticate header.
 *
 * @author Ulrich Mann
 *
 */
//@Provider
//@Priority(Priorities.AUTHENTICATION)
public abstract class BasicAuthenticationFilter implements ContainerRequestFilter {

    @Context
    ServletContext ctx;

    @Context
    HttpServletRequest httprequest;

    public static final String dummyOptionsUser = "dummyOptionsUser";

    @Override
    public void filter(ContainerRequestContext requestContext) throws IOException {

        try {
            this.filterBasic(requestContext);
        } catch (Exception e) {
            e.printStackTrace();
            refuseRequest403(requestContext);
        } finally {
        }
    }

    void filterBasic(ContainerRequestContext requestContext) throws Exception {

        System.out.println("Checking basic auth...");
//        Enumeration<String> e = httprequest.getHeaderNames();
//        while (e.hasMoreElements()) {
//            String el = e.nextElement();
//            System.out.println("Header " + el + " - " + httprequest.getHeader(el));
//        }

        UserCredentials credentials = null;

        String allowBasicAuth = this.ctx.getInitParameter("ALLOW_BASIC_AUTH");
        if (allowBasicAuth == null || !allowBasicAuth.equals("true")) {
            refuseRequest403(requestContext);
        } else {

            // TRY BASIC AUTHENTICATION
            credentials = AuthenticationFilterTools.getUserCredentials(this.httprequest);
            if (credentials != null) {
                this.authenticateBasic(requestContext, credentials);
            } else {
                System.out.println("No authentication header was set...");
                credentials = this.handleNoAuthenticationHeader(requestContext, this.httprequest);
                if (credentials == null) {
                    refuseRequest401(requestContext);
                    return;
                }
            }
        }

        if (!this.checkCredentials(requestContext, credentials)) {
            return;
        }
        System.out.println("BasicAuthenticationFilter, authentication successful for " + credentials.getUsername() + ":"
                + "password set=" + (credentials.getPassword() == null ? "no" : "yes"));

        this.prepareSecurityContext(requestContext, credentials);
    }

    private UserCredentials getDummyCredentialsForOptionsMethod() {
        return new UserCredentials(dummyOptionsUser, "dummyOptionsPassword");
    }

    void prepareSecurityContext(ContainerRequestContext requestContext, UserCredentials credentials) throws Exception {
        List<String> roles = this.getRoles();
        requestContext
                .setSecurityContext(new OverrideSecurityContext(credentials.getUsername(), roles, this.httprequest));
    }

    boolean checkCredentials(ContainerRequestContext requestContext, UserCredentials credentials) {
        if (credentials == null) {
            refuseRequest403(requestContext);
            return false;
        }
        if (credentials.getUsername().equals("") || credentials.getUsername() == null
                || credentials.getPassword() == null) {
            refuseRequest403(requestContext);
            return false;
        }
        return true;
    }

    /**
     * Authentication method which must be overridden by implementing filters. This
     * filter class should not be used for authentication itself.
     */
    public void authenticateBasic(ContainerRequestContext requestContext, UserCredentials credentials)
            throws Exception {
        throw new UnsupportedOperationException(
                "Invalid authentication filter base class. Use a custom filter implementation.");
    }

    /**
     * Handles a request, if no authorization header was sent with the request. If
     * overriding this method in a derived filter, we should always make sure to
     * return null credentials.
     *
     */
    public UserCredentials handleNoAuthenticationHeader(ContainerRequestContext crc, HttpServletRequest httpRequest)
            throws NotFoundException {
        if (crc.getMethod().equals("OPTIONS")) {
            // allows OPTIONS request
            System.out.println("OPTIONS method called. Allowing...");
            return this.getDummyCredentialsForOptionsMethod();
        }
        return null;
    }

    void init(FilterConfig filterConfig) throws ServletException {}

    /**
     * Reject an authentication attempt. Should be used with
     * includeBasicAuthHeader=true, when no credentials have been provided for the
     * authentication attempt.
     *
     * @param response
     */
    private static void refuseRequest(ContainerRequestContext requestContext, boolean issueAuthenticationHeader) {
        try {
            if (issueAuthenticationHeader) {
                String authVal = "None";
                authVal = "Basic auth=\"Internal\"";

                requestContext.abortWith(Response
                        .status(Response.Status.UNAUTHORIZED)
                        .header("WWW-Authenticate", authVal)
                        .type("text/plain")
                        .build());
                System.out.println("Responding 401. Requiring authentication details...");

            } else {
                requestContext.abortWith(Response
                        .status(Response.Status.FORBIDDEN)
                        .entity("Authentication failed. Access denied.")
                        .type("text/plain")
                        .build());
                System.out.println("Responding 403. User authentication failed.");
            }

        } catch (Exception e) {
            return;
        }
        return;
    }

    static void refuseRequest401(ContainerRequestContext requestContext) { refuseRequest(requestContext, true); }

    static void refuseRequest403(ContainerRequestContext requestContext) { refuseRequest(requestContext, false); }

    void destroy() {}

    /**
     * Simply returns default roles.
     * 
     * @param jdbcMeta
     * @param username
     * @return
     * @throws Exception
     */
    public ArrayList<String> getRoles() throws Exception {
        ArrayList<String> list = new ArrayList<String>();
        return list;
    }
}
