package de.narimo.geocore.ws.repository;

import java.io.IOException;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.ws.rs.NotFoundException;

import org.apache.commons.lang3.RandomStringUtils;

import de.narimo.commons.dto.RegistrationLanguage;
import de.narimo.commons.dto.User;
import de.narimo.commons.jdbc.JDBCConnectionJNDI;
import de.narimo.geocore.ws.registration.UserRegistrationDetails;

public class UserRepository {

    /**
     * This expects username to be unique in the system!
     *
     * @param username
     * @param jdbc
     * @return
     * @throws Exception
     */
    public static Integer getUserId(String username) {
        JDBCConnectionJNDI jdbc = null;
        try {
            jdbc = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            Integer userid = null;
            ResultSet rs = null;

            String idSql = "SELECT id FROM users WHERE username ILIKE ? AND enabled=true;";
            System.out.println(idSql);

            PreparedStatement ps = jdbc.prepareStatement(idSql);
            ps.setString(1, username);
            rs = ps.executeQuery();

            if (rs.next()) {
                userid = rs.getInt(1);
            }
            return userid;

        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException("Could not get userid for user " + username + ".");
        } finally {
            if (jdbc != null) {
                jdbc.closeAll();
                jdbc = null;
            }
        }
    }

    /**
     * This expects username to be unique in the system!
     *
     * @param username
     * @param jdbc
     * @return
     * @throws Exception
     */
    public static Optional<User> getUser(String username) {
        JDBCConnectionJNDI jdbc = null;
        try {
            if (username == null) {
                throw new IllegalArgumentException();
            }
            jdbc = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            String idSql = "SELECT id, username, email, language FROM users WHERE username ILIKE ? AND enabled=true;";

            PreparedStatement ps = jdbc.prepareStatement(idSql);
            ps.setString(1, username);
            ResultSet rs = ps.executeQuery();

            if (rs.next()) {
                User user = new User();
                user.setId(rs.getInt("id"));
                user.setEmail(rs.getString("email"));
                user.setName(rs.getString("username"));
                user.setLanguage(RegistrationLanguage.getRegistrationLanguage(rs.getString("language")));
                return Optional.ofNullable(user);
            }
            return Optional.empty();

        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException("Could not get user for username " + username + ".");
        } finally {
            if (jdbc != null) {
                jdbc.closeAll();
                jdbc = null;
            }
        }
    }

    /**
     * Creates a user and returns the alphanumeric secret that has been created for
     * the user to validate his account.
     * 
     * NOTE: This should work for a user, if his email is already registered but is
     * marked as disabled!
     *
     * @param email
     * @param organisation
     * @param userFirstName
     * @param userLastName
     * @param newPassword
     * @return
     * @throws IOException
     */
    public static String createUser(UserRegistrationDetails registrationDetails) throws IOException {

        JDBCConnectionJNDI jdbc = null;
        try {
            jdbc = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            String idSql = "INSERT INTO users (organisation, password, username, name, surname, email, regsecret, language, clientapp)"
                    + " VALUES(?, sha256g(?), ?, ?, ?, ?, ?, ?, ?)";
            System.out.println(idSql);

            String registrationSecret = createAlphanumericSecret();

            PreparedStatement ps = jdbc.prepareStatement(idSql);
            ps.setString(1, registrationDetails.getOrganisation());
            ps.setString(2, registrationDetails.getPassword());
            ps.setString(3, registrationDetails.getEmail());
            ps.setString(4, registrationDetails.getFirstName());
            ps.setString(5, registrationDetails.getLastName());
            ps.setString(6, registrationDetails.getEmail());
            ps.setString(7, registrationSecret);
            ps.setString(8, RegistrationLanguage.getRegistrationLanguage(registrationDetails.getLanguage()).toString());
            ps.setString(9, registrationDetails.getClientApp());

            ps.execute();

            return registrationSecret;

        } catch (SQLException e) {
            e.printStackTrace();
            if (e.getMessage().contains("duplicate key value violates unique constraint")) {
                throw new IOException(
                        "Username or email is already registered: " + registrationDetails.getEmail() + ".");
            } else if (e.getMessage().contains("null value in column")) {
                throw new IOException("Registration details not complete.");
            } else {
                throw new InternalError(e.getCause());
            }
        } catch (Exception e) {
            e.printStackTrace();
            throw new IOException("Could not create user " + registrationDetails.getEmail() + ".");
        } finally {
            if (jdbc != null) {
                jdbc.closeAll();
            }
            jdbc = null;
        }
    }

    /**
     * Return all users by the specified ids.
     *
     * @param adminIds
     * @return
     */
    public static List<User> getUsersById(List<Integer> userIds) {

        JDBCConnectionJNDI jdbcMeta = null;
        List<User> users = new ArrayList<User>();

        if (userIds == null || userIds.isEmpty()) {
            return users;
        }

        try {
            jdbcMeta = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            String sql = "SELECT id, email, username, language FROM users WHERE id IN (?) AND enabled=true;";
            String sqlIN = userIds.stream()
                    .map(x -> String.valueOf(x))
                    .collect(Collectors.joining(",", "(", ")"));
            sql = sql.replace("(?)", sqlIN);

            PreparedStatement ps0 = jdbcMeta.prepareStatement(sql);
            ResultSet rs = ps0.executeQuery();

            while (rs.next()) {
                User user = new User();
                user.setId(rs.getInt("id"));
                user.setEmail(rs.getString("email"));
                user.setName(rs.getString("username"));
                user.setLanguage(RegistrationLanguage.getRegistrationLanguage(rs.getString("language")));
                users.add(user);
            }
        } catch (Exception e) {
            e.printStackTrace();
            throw new InternalError();
        } finally {
            if (jdbcMeta != null) {
                jdbcMeta.closeAll();
                jdbcMeta = null;
            }
        }
        return users;
    }

    /**
     * Confirm a user registration in case a correct combination of email and
     * registration secret has been provided. This combination must be unique to be
     * accepted.
     *
     * @param email
     * @param registrationSecret
     * @throws IOException
     */
    public static void confirmUserRegistration(String email, String registrationSecret) throws IOException {
        JDBCConnectionJNDI jdbc = null;
        try {
            jdbc = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            String sql = "SELECT count(*) FROM users WHERE email ILIKE ? and regsecret=?;";

            PreparedStatement ps0 = jdbc.prepareStatement(sql);
            ps0.setString(1, email);
            ps0.setString(2, registrationSecret);
            ResultSet rs = ps0.executeQuery();

            int count = 0;
            if (rs.next()) {
                count = rs.getInt(1);
            }

            if (count != 1) {
                throw new IOException("Invalid registration details.");
            }

            String sql1 = "UPDATE users SET "
                    + "enabled=true, regsecret=null, confirmedat=now() "
                    + "WHERE email ILIKE ? and regsecret=?;";
            System.out.println(sql1);

            PreparedStatement ps = jdbc.prepareStatement(sql1);
            ps.setString(1, email);
            ps.setString(2, registrationSecret);
            ps.executeUpdate();

            ResultSet rs2 = ps0.executeQuery();

            int ccount = 0;
            if (rs2.next()) {
                ccount = rs2.getInt(1);
            }

            if (ccount != 0) {
                System.out.println("ccount is " + ccount + ".");
                throw new IOException("Confirmation could not be completed successfully.");
            }

        } catch (SQLException e) {
            e.printStackTrace();
            throw new InternalError(e.getCause());
        } catch (Exception e) {
            e.printStackTrace();
            throw new IOException("Could not confirm user registration" + email + ".");
        } finally {
            if (jdbc != null) {
                jdbc.closeAll();
                jdbc = null;
            }
        }
    }

    /**
     * Set the secret for resetting the password and the client app which was used
     * to issue the password reset request.
     * 
     * @param userid
     * @param clientapp
     * @return
     * @throws IOException
     */
    public static String setPasswordResetSecret(int userid, String clientapp) throws IOException {
        JDBCConnectionJNDI jdbc = null;
        try {
            jdbc = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            String secret = createAlphanumericSecret();

            String sql = "UPDATE users SET pwresetsecret=?, pwresetapp=? WHERE id=?;";

            PreparedStatement ps0 = jdbc.prepareStatement(sql);
            ps0.setString(1, secret);
            ps0.setString(2, clientapp);
            ps0.setInt(3, userid);
            int count = ps0.executeUpdate();

            if (count != 1) {
                throw new InternalError("Setting password reset secret failed.");
            }
            return secret;

        } catch (SQLException e) {
            e.printStackTrace();
            throw new InternalError(e.getCause());
        } catch (Exception e) {
            e.printStackTrace();
            throw new IOException("Setting password reset secret failed..");
        } finally {
            if (jdbc != null) {
                jdbc.closeAll();
                jdbc = null;
            }
        }
    }

    /**
     * Reset a password for a row that has been marked with a password reset secret
     * before.
     *
     * @param newPassword
     * @param passwordResetSecret
     * @throws IOException
     */
    public static String resetPassword(String newPassword, String passwordResetSecret, String email)
            throws IOException {
        JDBCConnectionJNDI jdbc = null;
        try {
            jdbc = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            String sql = "SELECT count(*) FROM users WHERE pwresetsecret=? AND email ILIKE ?;";

            PreparedStatement ps0 = jdbc.prepareStatement(sql);
            ps0.setString(1, passwordResetSecret);
            ps0.setString(2, email);
            ResultSet rs = ps0.executeQuery();

            int count = 0;
            if (rs.next()) {
                count = rs.getInt(1);
            }
            if (count != 1) {
                System.out.println("Reset password failed. There is no entry that matches "
                        + "mail " + email + " and secret " + passwordResetSecret + ".");
                throw new NotFoundException("Reset password failed.");
            }

            String sql1 = "UPDATE users SET password=sha256g(?), pwresetsecret=? WHERE pwresetsecret=? and email ILIKE ? RETURNING pwresetapp;";

            PreparedStatement ps1 = jdbc.prepareStatement(sql1);
            ps1.setString(1, newPassword);
            ps1.setString(2, null);
            ps1.setString(3, passwordResetSecret);
            ps1.setString(4, email);
            ResultSet rs1 = ps1.executeQuery();

            rs1.next();
            return rs1.getString("pwresetapp");

        } catch (SQLException e) {
            e.printStackTrace();
            throw new InternalError(e.getCause());
        } catch (Exception e) {
            e.printStackTrace();
            throw new IOException("Reset password failed.");
        } finally {
            if (jdbc != null) {
                jdbc.closeAll();
                jdbc = null;
            }
        }
    }

    public static void disableUser(int userid) {
        JDBCConnectionJNDI jdbc = null;
        try {
            jdbc = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            String removedUsername = "removed" + Math.round(Math.random() * 10000);
            String sql = "UPDATE users SET enabled=false, username=?, email=null, disabledat=now() WHERE id=?;";

            PreparedStatement ps0 = jdbc.prepareStatement(sql);
            ps0.setString(1, removedUsername);
            ps0.setInt(2, userid);
            ps0.executeUpdate();

        } catch (Exception e) {
            System.out.println("Updating user failed.");
            e.printStackTrace();
            throw new InternalError("User account should have been deleted but might have failed!");
        } finally {
            if (jdbc != null) {
                jdbc.closeAll();
                jdbc = null;
            }
        }
    }

    public static void saveUser(int userid, RegistrationLanguage registrationLanguage) {
        JDBCConnectionJNDI jdbc = null;
        try {
            jdbc = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            String sql = "UPDATE users SET language=? WHERE id=?;";

            PreparedStatement ps0 = jdbc.prepareStatement(sql);
            ps0.setString(1, registrationLanguage.toString());
            ps0.setInt(2, userid);
            ps0.executeUpdate();

        } catch (Exception e) {
            System.out.println("Updating user failed.");
            e.printStackTrace();
            throw new InternalError("Updating user failed.");
        } finally {
            if (jdbc != null) {
                jdbc.closeAll();
            }
        }
    }

    public static void incrementAllowedWorkspaceCount(User user) {
        JDBCConnectionJNDI jdbc = null;
        try {
            jdbc = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            String sql = "UPDATE users SET allowedworkspacecount=allowedworkspacecount+1 WHERE id=" + user.getId()
                    + ";";
            jdbc.execute(sql);

        } catch (Exception e) {
            System.out.println("Updating user failed.");
            e.printStackTrace();
            throw new InternalError("Updating user failed.");
        } finally {
            if (jdbc != null) {
                jdbc.closeAll();
            }
        }
    }

    private static String createAlphanumericSecret() { return RandomStringUtils.randomAlphanumeric(128); }
}
