package de.narimo.georepo.server.repository;

import java.io.IOException;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.ws.rs.NotFoundException;

import org.geotools.feature.FeatureCollection;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.Point;
import org.opengis.feature.Property;
import org.opengis.feature.simple.SimpleFeature;

import de.narimo.commons.jdbc.JDBCConnectionJNDI;
import de.narimo.georepo.server.GeorepoDatasource;
import de.narimo.georepo.server.db.GeorepoConstants;
import de.narimo.georepo.server.dto.DiffDataset;
import de.narimo.georepo.server.layer.FeatureBuilder;
import de.narimo.georepo.server.tools.QueryCheck;

public class DatasetRepository {

    public static String authTableSuffix = "_auth";

    /**
     * Insert new datasets into a diff table.
     *
     * @param tableName
     *            the name of a diff table
     * @param features
     */
    public static void insertDatasets(String diffTableName, String authTableName, List<SimpleFeature> features, long userId)
            throws IOException {

        QueryCheck.checkTableName(diffTableName);
        QueryCheck.checkDiffTableName(diffTableName);
        QueryCheck.checkTableName(authTableName);

        JDBCConnectionJNDI jdbcData = null;

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            Map<String, String> columnsAndTypes = DatasetRepository.getColumnTypes(diffTableName,
                    GeorepoConstants.DEFAULTSCHEMA);
            columnsAndTypes.put(GeorepoConstants.GRPSTATUSCOLUMN, "text");
            columnsAndTypes.put(GeorepoConstants.GEOMETRYCOLUMN, "geom");

            String insert = createFeatureInsertUpdateString(diffTableName, columnsAndTypes, features, 'n');

            ResultSet rs = jdbcData.executeQuery(insert);

            List<Integer> insertedGrpfids = new ArrayList<>();
            while (rs.next()) {
                insertedGrpfids.add(rs.getInt(GeorepoConstants.GRPFIDCOLUMN));
            }

            String auth = createAuthTableUpdateString(insertedGrpfids, userId, authTableName);
            jdbcData.execute(auth);

        } catch (SQLException e) {
            e.printStackTrace();
        } catch (Exception e) {
            throw new IOException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    /**
     * Insert datasets into a diff table that declare updates from users.
     *
     * @param diffTableName
     * @param features
     */
    public static void updateDatasets(String diffTableName, String authTableName, List<SimpleFeature> features, long userId)
            throws IOException {

        QueryCheck.checkTableName(diffTableName);
        QueryCheck.checkDiffTableName(diffTableName);
        QueryCheck.checkTableName(authTableName);

        JDBCConnectionJNDI jdbcData = null;

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            Map<String, String> columnsAndTypes = DatasetRepository.getColumnTypes(diffTableName,
                    GeorepoConstants.DEFAULTSCHEMA);
            columnsAndTypes.put(GeorepoConstants.GRPSTATUSCOLUMN, "text");
            columnsAndTypes.put(GeorepoConstants.GEOMETRYCOLUMN, "geom");

            String update = createFeatureInsertUpdateString(diffTableName, columnsAndTypes, features, 'm');

            ResultSet rs = jdbcData.executeQuery(update);

            List<Integer> insertedGrpfids = new ArrayList<>();
            while (rs.next()) {
                insertedGrpfids.add(rs.getInt(GeorepoConstants.GRPFIDCOLUMN));
            }

            String auth = createAuthTableUpdateString(insertedGrpfids, userId, authTableName);
            jdbcData.execute(auth);

        } catch (SQLException e) {
            e.printStackTrace();
        } catch (Exception e) {
            throw new IOException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    public static void deleteDatasets(String diffTableName, String authTableName, List<SimpleFeature> features, long userId)
            throws IOException {

        QueryCheck.checkTableName(diffTableName);
        QueryCheck.checkDiffTableName(diffTableName);
        QueryCheck.checkTableName(authTableName);

        JDBCConnectionJNDI jdbcData = null;

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            Map<String, String> columnsAndTypes = DatasetRepository.getColumnTypes(diffTableName,
                    GeorepoConstants.DEFAULTSCHEMA);
            columnsAndTypes.put(GeorepoConstants.GRPSTATUSCOLUMN, "text");

            String delete = createFeatureDeleteString(diffTableName, features);
            ResultSet rs = jdbcData.executeQuery(delete);

            List<Integer> insertedGrpfids = new ArrayList<>();
            while (rs.next()) {
                insertedGrpfids.add(rs.getInt(GeorepoConstants.GRPFIDCOLUMN));
            }

            String auth = createAuthTableUpdateString(insertedGrpfids, userId, authTableName);
            jdbcData.execute(auth);

        } catch (SQLException e) {
            e.printStackTrace();
        } catch (Exception e) {
            throw new IOException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    /**
     * Add, update or remove a dataset from an original data table.
     *
     * @param diffTable
     * @param dataTable
     * @param grpfid
     * @param features
     * @param userId
     * @throws IOException
     */
    public static void acceptDataset(String diffTable, String dataTable, String authTable, int grpfid) throws IOException {

        QueryCheck.checkTableName(diffTable);
        QueryCheck.checkTableName(dataTable);
        QueryCheck.checkTableName(authTable);
        QueryCheck.checkDiffTableName(diffTable);

        JDBCConnectionJNDI jdbcData = null;

        try {

            try {
                jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");
            } catch (Exception e) {
                throw new SQLException(e);
            }
            Map<String, String> columnsAndTypes = DatasetRepository.getColumnTypes(diffTable,
                    GeorepoConstants.DEFAULTSCHEMA);
            columnsAndTypes.put(GeorepoConstants.GEOMETRYCOLUMN, "geom");

            List<String> columns = new ArrayList<>(columnsAndTypes.keySet());
            String accept = createAcceptFeatureString(dataTable, diffTable, authTable, grpfid, columns);
            System.out.println(accept);

            try {
                jdbcData.execute(accept);
            } catch (Exception e) {
                throw new SQLException(e);
            }

        } catch (SQLException e) {
            e.printStackTrace();
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    public static void declineDataset(String diffTable, String authTable, int grpfid) throws IOException {

        QueryCheck.checkTableName(diffTable);
        QueryCheck.checkTableName(authTable);
        QueryCheck.checkDiffTableName(diffTable);

        JDBCConnectionJNDI jdbcData = null;
        try {

            try {
                jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");
            } catch (Exception e) {
                throw new SQLException(e);
            }

            String decline = createDeclineChangesetString(diffTable, authTable, grpfid);
            System.out.println(decline);

            try {
                jdbcData.execute(decline);
            } catch (Exception e) {
                throw new SQLException(e);
            }

        } catch (SQLException e) {
            e.printStackTrace();
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    /**
     * Creates the SQL string to insert feature updates.
     *
     * @param tableName
     * @param columnsAndTypes
     * @param features
     * @param status
     * @return
     * @throws IOException
     */
    private static String createFeatureInsertUpdateString(String tableName, Map<String, String> columnsAndTypes,
            List<SimpleFeature> features, char status) throws IOException {

        QueryCheck.checkTableName(tableName);

        if (!GeorepoConstants.ALLOWEDUPDATESTATI.contains(String.valueOf(status))) {
            throw new IllegalArgumentException("Unknown update status: " + status);
        }
        String columnNames = String.join(",", columnsAndTypes.keySet());

        StringBuilder b = new StringBuilder();
        b.append("INSERT INTO public." + tableName + "(" + columnNames + ") VALUES ");

        int j = 0;
        for (SimpleFeature feature : features) {

            // has property gfid to reference real world feature?
            boolean hasGfid = feature.getProperty(GeorepoConstants.GFIDCOLUMN) != null;

            if (status == 'm') {
                if (!hasGfid) {
                    throw new IllegalArgumentException(
                            "At least one feature is missing required property " + GeorepoConstants.GFIDCOLUMN + ".");
                }
            } else if (status == 'n') {
                if (hasGfid) {
                    throw new IllegalArgumentException(
                            "Features may not specify property " + GeorepoConstants.GFIDCOLUMN + ".");
                }
            }

            if (j > 0) {
                b.append(",");
            }
            b.append("(");
            int i = 0;

            Geometry geom = (Geometry) feature.getDefaultGeometry();
            String geomType = geom.getGeometryType();

            for (String column : columnsAndTypes.keySet()) {
                if (i > 0) {
                    b.append(",");
                }
                if (column.equals(GeorepoConstants.GRPSTATUSCOLUMN)) {
                    b.append("'" + status + "'");

                } else if (column.equals(GeorepoConstants.GEOMETRYCOLUMN)
                        && geomType.equals("Point")) {
                    // supports point geometries currently
                    double lon = geom.getCoordinates()[0].getX();
                    double lat = geom.getCoordinates()[0].getY();
                    b.append("ST_SetSRID(ST_MakePoint("
                            + "cast(" + lon + " as float), "
                            + "cast(" + lat + " as float)), "
                            + GeorepoConstants.allowedSRID + ")");
                } else {
                    Property p = feature.getProperty(column);
                    if (p != null) {
                        if (columnsAndTypes.get(column).equals("text")
                                || columnsAndTypes.get(column).equals("character varying")) {
                            // character types
                            b.append("'" + feature.getProperty(column).getValue() + "'");
                        } else {
                            // numeric types
                            b.append(feature.getProperty(column).getValue());
                        }
                    } else {
                        b.append("null");
                    }
                }
                i++;
            }
            b.append(")");
            j++;
        }
        b.append(" RETURNING " + GeorepoConstants.GRPFIDCOLUMN + ";");

        System.out.println("Inserting datasets: " + b.toString());

        return b.toString();
    }

    /**
     * Creates a sql query string to insert a delete feature changeset to a diff table.
     *
     * @param diffTableName
     * @param features
     * @return
     * @throws IOException
     */
    private static String createFeatureDeleteString(String diffTableName, List<SimpleFeature> features) throws IOException {
        char status = 'd';

        QueryCheck.checkTableName(diffTableName);
        QueryCheck.checkDiffTableName(diffTableName);

        if (!GeorepoConstants.ALLOWEDUPDATESTATI.contains(String.valueOf(status))) {
            throw new IllegalArgumentException("Unknown update status: " + status);
        }

        StringBuilder b = new StringBuilder();
        b.append("INSERT INTO public." + diffTableName + "("
                + GeorepoConstants.GFIDCOLUMN + "," + GeorepoConstants.GRPSTATUSCOLUMN + ") VALUES ");

        int j = 0;
        for (SimpleFeature feature : features) {

            // has property gfid to reference real world feature?
            boolean hasGfid = feature.getProperty(GeorepoConstants.GFIDCOLUMN) != null;

            if (!hasGfid) {
                throw new IllegalArgumentException(
                        "At least one feature is missing required property " + GeorepoConstants.GFIDCOLUMN + ".");
            }

            if (j > 0) {
                b.append(",");
            }
            b.append("(");
            b.append("'" + feature.getProperty(GeorepoConstants.GFIDCOLUMN).getValue() + "'");
            b.append(",");
            b.append("'" + status + "'");
            b.append(")");

            j++;
        }
        b.append(" RETURNING " + GeorepoConstants.GRPFIDCOLUMN + ";");

        return b.toString();
    }

    /**
     * Creates a sql query string to update insert a new auth dataset.
     *
     * @param grpfids
     * @param userid
     * @param authTableName
     * @return
     * @throws IOException
     */
    private static String createAuthTableUpdateString(List<Integer> grpfids, long userid, String authTableName)
            throws IOException {

        QueryCheck.checkTableName(authTableName);

        StringBuilder b = new StringBuilder();
        b.append("INSERT INTO public." + authTableName + " (" + GeorepoConstants.GRPFIDCOLUMN
                + ",userid) VALUES ");

        int i = 0;
        for (int grpfid : grpfids) {
            if (i > 0) {
                b.append(",");
            }
            b.append("(" + grpfid + "," + userid + ")");
            i++;
        }
        b.append(";");
        System.out.println(b.toString());
        return b.toString();
    }

    /**
     * Creates a sql query string to copy a changeset from a diff table to an
     * original data table.
     *
     * @param tableName
     * @param columnsAndTypes
     * @param features
     * @param status
     * @return
     * @throws SQLException
     *             when the columns differ from the original data table
     */
    private static String createAcceptFeatureString(
            String dataTable, String diffTable, String authTable, int grpfid, List<String> changesetColumns)
            throws NotFoundException, IllegalArgumentException, IOException {

        QueryCheck.checkTableName(dataTable);
        QueryCheck.checkTableName(diffTable);
        QueryCheck.checkTableName(authTable);
        QueryCheck.checkDiffTableName(diffTable);

        DiffDataset diffDataset = getDiffDatasetProperties(diffTable, grpfid);
        if (diffDataset == null) {
            throw new NotFoundException("No changeset with id " + grpfid + " exists;");
        }
        char status = diffDataset.getGrpstatus().charAt(0);
        Integer gfid = diffDataset.getGfid();

        System.out.println("Changeset to merge: status: " + status + ", gfid: " + gfid);

        changesetColumns.remove(GeorepoConstants.GRPSTATUSCOLUMN);
        changesetColumns.remove(GeorepoConstants.GRPFIDCOLUMN);
        changesetColumns.remove(GeorepoConstants.GFIDCOLUMN);

        StringBuilder b = new StringBuilder();
        if (status == 'm' && gfid != null) {
            b.append("UPDATE public." + dataTable + " o SET ");

            int c = 0;
            for (String column : changesetColumns) {
                if (column.equals(GeorepoConstants.GRPSTATUSCOLUMN)
                        || column.equals(GeorepoConstants.GRPFIDCOLUMN)
                        || column.equals(GeorepoConstants.GFIDCOLUMN)) {
                    continue;
                }
                if (c > 0) {
                    b.append(", ");
                }

                b.append(column + "=d." + column);
                c++;
            }
            b.append(" FROM " + diffTable + " d WHERE o.gfid=d.gfid");
            // AND d.gfid = 1
            b.append(" AND d.grpfid=" + grpfid + ";");

        } else if (status == 'n' && gfid == null) {
            String columnNames = String.join(",", changesetColumns);
            b.append("INSERT INTO public." + dataTable + "(" + columnNames + ") ");
            b.append("SELECT " + columnNames + " FROM " + diffTable + " ");
            b.append("WHERE " + GeorepoConstants.GRPFIDCOLUMN + "=" + grpfid + ";");

        } else if (status == 'd' && gfid != null) {
            // gfid is unique in orignal data table
            b.append("DELETE FROM public." + dataTable + " "
                    + "WHERE " + GeorepoConstants.GFIDCOLUMN + "=" + gfid + ";");
        } else {
            throw new IllegalArgumentException("Unsupported status " + status + " or corrupt changeset.");
        }

        String cleanupDiffChangesetSql = " DELETE FROM public." + diffTable + " WHERE " + GeorepoConstants.GRPFIDCOLUMN + "="
                + grpfid + ";";
        b.append(cleanupDiffChangesetSql);

        String cleanupAuthChangesetSql = " DELETE FROM public." + authTable + " WHERE " + GeorepoConstants.GRPFIDCOLUMN + "="
                + grpfid + ";";
        b.append(cleanupAuthChangesetSql);

        if (status == 'd' && gfid != null) {
            // If we accept a DELETE changeset, we have to remove
            // all remaining changesets for the same gfid as well
            // to prevent inconsistencies
            String cleanupAllRelatedDiffChangesetsSql = " DELETE FROM public." + diffTable + " WHERE "
                    + GeorepoConstants.GFIDCOLUMN + "=" + gfid + ";";
            b.append(cleanupAllRelatedDiffChangesetsSql);

            String cleanupAllRelatedAuthChangesetsSql = " DELETE FROM public." + authTable
                    + " WHERE " + GeorepoConstants.GFIDCOLUMN + "=" + gfid + ";";
            b.append(cleanupAllRelatedAuthChangesetsSql);
        }
        return b.toString();
    }

    private static String createDeclineChangesetString(
            String diffTable, String authTable, int grpfid) throws NotFoundException, IllegalArgumentException, IOException {

        QueryCheck.checkTableName(diffTable);
        QueryCheck.checkTableName(authTable);
        QueryCheck.checkDiffTableName(diffTable);

        DiffDataset diff = getDiffDatasetProperties(diffTable, grpfid);
        if (diff == null) {
            throw new NotFoundException("No changeset with id " + grpfid + " exists;");
        }

        StringBuilder b = new StringBuilder();
        b.append("DELETE FROM public." + diffTable + " WHERE " + GeorepoConstants.GRPFIDCOLUMN + "="
                + grpfid + ";");
        b.append("DELETE FROM public." + authTable + " WHERE " + GeorepoConstants.GRPFIDCOLUMN + "="
                + grpfid + ";");
        return b.toString();
    }

    private static DiffDataset getDiffDatasetProperties(String diffTable, int grpfid) throws IOException {

        QueryCheck.checkTableName(diffTable);
        QueryCheck.checkDiffTableName(diffTable);

        JDBCConnectionJNDI jdbcData = null;

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql = "SELECT * FROM public." + diffTable + " WHERE " +
                    GeorepoConstants.GRPFIDCOLUMN + "=?;";

            PreparedStatement ps = jdbcData.prepareStatement(sql);
            ps.setInt(1, grpfid);

            ResultSet rs = ps.executeQuery();
            if (rs.next()) {
                DiffDataset diff = new DiffDataset();
                diff.setGfid(rs.getInt(GeorepoConstants.GFIDCOLUMN));
                if (rs.wasNull()) {
                    // result set returns 0 for integer null values
                    diff.setGfid(null);
                }
                diff.setGrpfid(rs.getInt(GeorepoConstants.GRPFIDCOLUMN));
                diff.setGrpstatus(rs.getString(GeorepoConstants.GRPSTATUSCOLUMN));
                return diff;
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
        return null;
    }

    public static void createDatasetAuthTable(int layerIndex) {
        JDBCConnectionJNDI jdbcData = null;

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql = "CREATE TABLE IF NOT EXISTS public." + GeorepoDatasource.georepoDatasourcePrefix + ""
                    + layerIndex + authTableSuffix
                    + "("
                    + "id SERIAL PRIMARY KEY,"
                    + "grpfid integer UNIQUE NOT NULL,"
                    + "userid integer NOT NULL"
                    + ");";
            System.out.println(sql);

            jdbcData.execute(sql);

        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    private static Map<String, String> getColumnTypes(String tableName, String schemaName) {
        JDBCConnectionJNDI jdbcData = null;

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql0 = "SELECT column_name, data_type FROM information_schema.columns "
                    + "WHERE table_schema = ? "
                    + "AND table_name   = ?;";

            PreparedStatement ps0 = jdbcData.prepareStatement(sql0);
            ps0.setString(1, schemaName);
            ps0.setString(2, tableName);

            List<String> reservedColumnNames = new ArrayList<>();
            reservedColumnNames.add(GeorepoConstants.GRPFIDCOLUMN);
            // reservedColumnNames.add(GeorepoConstants.GFIDCOLUMN);
            reservedColumnNames.add(GeorepoConstants.GEOMETRYCOLUMN);
            reservedColumnNames.add(GeorepoConstants.GRPSTATUSCOLUMN);

            Map<String, String> columnNames = new HashMap<>();

            ResultSet rs0 = ps0.executeQuery();
            while (rs0.next()) {
                String columnName = rs0.getString("column_name");
                if (!reservedColumnNames.contains(columnName)) {
                    columnNames.put(columnName, rs0.getString("data_type"));
                }
            }
            return columnNames;
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
        return null;
    }

    public static FeatureCollection getDatasets(String tableName, long userId) throws IOException {

        QueryCheck.checkTableName(tableName);

        JDBCConnectionJNDI jdbcData = null;

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            Map<String, String> columnsAndTypes = DatasetRepository.getColumnTypes(tableName,
                    GeorepoConstants.DEFAULTSCHEMA);

            Map<String, Class> propertyTypes = new HashMap<String, Class>();

            for (String colName : columnsAndTypes.keySet()) {
                String colType = columnsAndTypes.get(colName);
                if (colType.equals("integer")) {
                    propertyTypes.put(colName, Integer.class);
                } else if (colType.equals("double precision")) {
                    propertyTypes.put(colName, Double.class);
                } else {
                    propertyTypes.put(colName, String.class);
                }
            }

            String query = getDatasetsQuery(tableName);
            ResultSet rs = jdbcData.executeQuery(query);

            FeatureBuilder fb = new FeatureBuilder(propertyTypes);
            GeometryFactory gf = new GeometryFactory();

            List<SimpleFeature> simpleFeatures = new ArrayList<>();

            while (rs.next()) {
                double x = rs.getDouble("st_x");
                double y = rs.getDouble("st_y");

                Geometry geometry = null;
                if (!rs.wasNull()) {
                    geometry = createPointGeometry(x, y, gf);
                }
                Map<String, Object> properties = new HashMap<>();
                for (String colName : columnsAndTypes.keySet()) {
                    properties.put(colName, rs.getString(colName));
                }

                SimpleFeature simpleFeature = fb.createSimpleFeature(geometry, properties);
                simpleFeatures.add(simpleFeature);
            }
            return fb.createFeatureCollection(simpleFeatures);

        } catch (SQLException e) {
            e.printStackTrace();
        } catch (Exception e) {
            throw new IOException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
        return null;
    }

    public static FeatureCollection getModifiedDatasets(String diffTableName, String authTableName, long userId, boolean asAdmin)
            throws IOException {

        QueryCheck.checkTableName(diffTableName);
        QueryCheck.checkTableName(authTableName);

        JDBCConnectionJNDI jdbcData = null;

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            Map<String, String> columnsAndTypes = DatasetRepository.getColumnTypes(diffTableName,
                    GeorepoConstants.DEFAULTSCHEMA);
            columnsAndTypes.put(GeorepoConstants.GRPSTATUSCOLUMN, "text");
            columnsAndTypes.put(GeorepoConstants.GRPFIDCOLUMN, "int");

            Map<String, Class> propertyTypes = new HashMap<String, Class>();

            for (String colName : columnsAndTypes.keySet()) {
                String colType = columnsAndTypes.get(colName);
                if (colType.equals("integer")) {
                    propertyTypes.put(colName, Integer.class);
                } else if (colType.equals("double precision")) {
                    propertyTypes.put(colName, Double.class);
                } else {
                    propertyTypes.put(colName, String.class);
                }
            }

            String query = getModifiedDatasetsQuery(diffTableName, authTableName, userId, asAdmin);
            ResultSet rs = jdbcData.executeQuery(query);

            FeatureBuilder fb = new FeatureBuilder(propertyTypes);
            GeometryFactory gf = new GeometryFactory();

            List<SimpleFeature> simpleFeatures = new ArrayList<>();

            while (rs.next()) {
                double x = rs.getDouble("st_x");
                double y = rs.getDouble("st_y");

                Geometry geometry = null;
                if (!rs.wasNull()) {
                    geometry = createPointGeometry(x, y, gf);
                }
                Map<String, Object> properties = new HashMap<>();
                for (String colName : columnsAndTypes.keySet()) {
                    properties.put(colName, rs.getString(colName));
                }

                SimpleFeature simpleFeature = fb.createSimpleFeature(geometry, properties);
                simpleFeatures.add(simpleFeature);
            }
            return fb.createFeatureCollection(simpleFeatures);

        } catch (SQLException e) {
            e.printStackTrace();
        } catch (Exception e) {
            throw new IOException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
        return null;
    }

    /**
     * Create a single point geometry instance.
     *
     * @param x
     * @param y
     * @param gf
     * @return
     */
    private static Geometry createPointGeometry(double x, double y, GeometryFactory gf) {
        Coordinate coordinate = new Coordinate(x, y);
        Point point = gf.createPoint(coordinate);
        return gf.createGeometry(point);
    }

    private static String getDatasetsQuery(String tableName) {
        return "SELECT ST_X(the_geom), ST_Y(the_geom), * from public." + tableName + ";";
    }

    /**
     * Returns the sql query string to use for retrieving modified features. If asAdmin is true, the query will featch all
     * features from the diff table. Otherwise it fetches only those datasets, that are created by the requesting user.
     *
     * @param diffTableName
     * @param userId
     * @param asAdmin
     * @return
     * @throws IOException
     */
    private static String getModifiedDatasetsQuery(String diffTableName, String authTableName, long userId, boolean asAdmin)
            throws IOException {
        QueryCheck.checkTableName(diffTableName);
        QueryCheck.checkTableName(authTableName);

        if (asAdmin) {
            return "SELECT ST_X(the_geom), ST_Y(the_geom), * from public." + diffTableName + ";";
        } else {
            return "SELECT ST_X(the_geom), ST_Y(the_geom), * from public." + diffTableName + " d JOIN public."
                    + authTableName + " a "
                    + "ON d." + GeorepoConstants.GRPFIDCOLUMN + " = a." + GeorepoConstants.GRPFIDCOLUMN + " "
                    + "WHERE a.userid=" + userId + ";";
        }
    }

    public static int getLayerId(String workspace, String featureType) {

        JDBCConnectionJNDI jdbcMeta = null;

        try {
            jdbcMeta = new JDBCConnectionJNDI("jdbc/georepoMetaResource");

            String sql = "SELECT id from public.datasets WHERE workspace = ? AND featuretype = ?;";
            System.out.println(sql);

            PreparedStatement ps = jdbcMeta.prepareStatement(sql);
            ps.setString(1, workspace);
            ps.setString(2, featureType);

            ResultSet rs = ps.executeQuery();

            int count = 0;
            Integer id = null;
            while (rs.next()) {
                id = rs.getInt("id");
                count++;
            }

            if (count > 1) {
                throw new NotFoundException(
                        "Could not determine distinct layer id. Please review datasets table consistency for workspace "
                                + workspace + " and feature type " + featureType + "!");
            }

            return id;
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (jdbcMeta != null) {
                jdbcMeta.closeAll();
                jdbcMeta = null;
            }
        }
        throw new NotFoundException("Could not determine layer id.");
    }
}
