package de.narimo.georepo.server.api;

import java.io.IOException;
import java.sql.SQLException;

import javax.servlet.ServletContext;
import javax.ws.rs.ForbiddenException;
import javax.ws.rs.OPTIONS;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.SecurityContext;
import javax.ws.rs.ext.Provider;

import de.narimo.commons.dto.geometa.User;
import de.narimo.geocore.ws.repository.UserRepository;
import de.narimo.georepo.server.db.GeorepoConstants;
import de.narimo.georepo.server.geoserver.GeoserverLayer;
import de.narimo.georepo.server.geoserver.GeoserverWorkspace;
import de.narimo.georepo.server.tools.AdminTools;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;

@Provider
@Path("/workspaces")
@Api(value = "WorkspacesController")
public class WorkspacesController {

    /**
     * Allows a browser to issue a CORS preflight request.
     * 
     * Generic response for OPTIONS request currently does not support setting and CORS headers.
     *
     * Application-specific OPTIONS responses may be set by the application itself or at sub-paths.
     *
     * @param sec
     * @param ctx
     * @return
     * @throws Exception
     */
    @OPTIONS
    @Path("")
    public static Response optionsLayerRequest(
            @Context SecurityContext sec,
            @Context ServletContext ctx) throws Exception {
        return Response.ok().build();
    }

    @POST
    @Path("/")
    @ApiOperation(value = "Creates a new workspace")
    public static Response createWorkspace(@Context SecurityContext sec, @Context ServletContext ctx) throws Exception {

        String username = sec.getUserPrincipal().getName();
        User user = UserRepository.getUser(username).get();

        String workspaceName = null;
        try {
            workspaceName = createNewWorkspace(ctx, user);
        } catch (ForbiddenException e) {
            System.out.println("User " + username + " has no permission to create workspace.");
            return Response.status(403).entity("User has no permission to create workspace.").build();
        } catch (IOException ioe) {
            System.out.println("An issue occured while creating workspace.");
            ioe.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }

        return Response.status(201)
                .entity("Workspace " + workspaceName + " has been created. "
                        + "Please contact customer service to enable its web services.")
                .build();
    }

    /**
     * Creates a workspace with a number that is not yet taken.
     * 
     * @param ctx
     * @param user
     * @return
     * @throws IOException
     * @throws SQLException
     */
    public static String createNewWorkspace(ServletContext ctx, User user) throws IOException, SQLException {

        if (user.getId() == null) {
            throw new IllegalArgumentException("Could not create a workspace for user " + user.getName()
                    + ". Unknown user " + user.getName() + ".");
        }

        AdminTools.checkCanCreateWorkspace(user.getId());

        String newWorkspace = getNewWorkspaceName();
        GeoserverLayer layer = new GeoserverLayer(ctx);
        while (layer.workspaceExists(layer.getWorkspacesXML(), newWorkspace)) {
            // re-create a workspace name which is not yet taken
            newWorkspace = getNewWorkspaceName();
        }

        GeoserverWorkspace ws = new GeoserverWorkspace(newWorkspace, layer.getGeoserverRestUrl(), layer.getHeaders(),
                layer.getGeoserverUser(), layer.getGeoserverPass());
        ws.createWorkspace();

        // Changing workspace settings for a freshly created workspace always provokes an 405 error.
        // boolean transactionSupport = false;
        // if (!transactionSupport) {
        // ws.setTransactional(false, newWorkspace);
        // }

        // Do it separate. Workspace and services must be enabled manually
        // first!
        // ws.modifyWorkspaceWMSSettings(workspace, title, description,
        // maintainer, fees, accessConstraints, onlineResource, true);

        // Set permissions as workspace owner
        // setWorkspacePermissions0(workspace, user.getEmail(), "rw");

        return newWorkspace;
    }

    public static String getNewWorkspaceName() {
        long workspaceCount = Math.round(Math.random() * 10000);
        return GeorepoConstants.WORKSPACEPREFIX + workspaceCount;
    }
}
