package de.narimo.georepo.server.geoserver;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.simpleframework.xml.Serializer;
import org.simpleframework.xml.core.Persister;

import com.fasterxml.jackson.databind.JsonMappingException;

import de.narimo.commons.http.URLResponse;
import de.narimo.commons.json.JsonConverter;
import de.narimo.commons.ws.http.HttpMethod;
import de.narimo.commons.ws.http.HttpURLClient;
import de.narimo.georepo.server.dto.DataStore;
import de.narimo.georepo.server.dto.DataStoreXML;
import de.narimo.georepo.server.dto.DataStores;
import de.narimo.georepo.server.dto.DataStoresList;

public class GeoserverWorkspace {

    private String geoserverRestUrl;
    private Map<String, String> headers;
    private String geoserverAdmin;
    private String geoserverPass;

    private String workspace;

    public GeoserverWorkspace(
            String geoserverRestUrl,
            Map<String, String> headers,
            String geoserverAdmin,
            String geoserverPass) {

        this.geoserverRestUrl = geoserverRestUrl;
        this.headers = headers;
        this.geoserverAdmin = geoserverAdmin;
        this.geoserverPass = geoserverPass;
    }

    public GeoserverWorkspace(
            String workspaceName,
            String geoserverRestUrl,
            Map<String, String> headers,
            String geoserverAdmin,
            String geoserverPass) {

        this.workspace = workspaceName;
        this.geoserverRestUrl = geoserverRestUrl;
        this.headers = headers;
        this.geoserverAdmin = geoserverAdmin;
        this.geoserverPass = geoserverPass;
    }

    /**
     * Creates a workspace by creating a namespace which auto-creates a bare
     * workspace.
     *
     * @param workspaceName
     * @throws Exception
     */
    public void createWorkspace() throws IOException {
        this.createNamespace(this.workspace);

        System.out.println("New workspace " + this.workspace + " has been created but cannot be activated. "
                + "Please go to admin interface and enable this workspace and its services.");
    }

    /**
     * Create a new namespace. This will automatically create a workspace too.
     *
     * @param namespace
     * @param nsUri
     * @throws Exception
     */
    public void createNamespace(String namespace) throws IOException {

        if (namespace == null) {
            throw new IllegalArgumentException("Cannot create namespace null.");
        }

        String namespaceUri = this.geoserverRestUrl + "/namespaces/" + namespace;

        String body = "<namespace><prefix>" + namespace + "</prefix><uri>" + namespaceUri + "</uri></namespace>";

        URLResponse rr = null;
        try {
            rr = HttpURLClient.sendRequest(
                    this.geoserverRestUrl + "/namespaces",
                    null,
                    this.headers,
                    body,
                    HttpMethod.POST,
                    null,
                    this.geoserverAdmin,
                    this.geoserverPass,
                    false,
                    false);

        } catch (Exception e) {
            e.printStackTrace();
            throw new IOException(e);
        }
        if (rr.getStatusCode() < 200 || rr.getStatusCode() > 201) {
            throw new InternalError(
                    "Creation of namespace " + namespace + " failed with status " + rr.getStatusCode() + ".");
        }

    }

    /**
     * TODO: Seems to work basically but active is not effective.
     * According to this
     * https://gis.stackexchange.com/questions/109226/how-to-set-up-a-workspace-setting-in-geoserver-using-curl
     * and this
     * https://gis.stackexchange.com/questions/244029/creating-workspace-data-store-and-layer-via-geoserver-rest-api
     * there's no option to enable a created workspace using the API right now.
     *
     * We have to do this manually...
     *
     * @param workspaceName
     * @throws Exception
     */
    public void modifyWorkspaceWMSSettings(
            String workspaceName,
            String title,
            String description,
            String maintainer,
            String fees,
            String accessConstraints,
            String onlineResource,
            boolean enabled) throws Exception {

        String body = "<wms>"
                + "<workspace>"
                + "	<name>" + workspaceName + "</name>"
                + "</workspace>"
                + "<enabled>" + enabled + "</enabled>"
                // + "<name>WMS</name>" //do not change WMS name!
                + "<title>" + title + "</title>"
                + "<maintainer>" + maintainer + "</maintainer>"
                + "<abstrct>" + description + "</abstrct>"
                + "<accessConstraints>" + accessConstraints + "</accessConstraints>"
                + "<fees>" + fees + "</fees>"
                + "<keywords>"
                // + "<string>WFS</string>"
                + "<string>WMS</string>"
                // + "<string>GEOSERVER</string>"
                + "</keywords>"
                // + "<metadataLink/>
                // + "<citeCompliant>false</citeCompliant>
                + "<onlineResource>" + onlineResource + "</onlineResource>"
                // + "<schemaBaseURL>http://schemas.opengis.net</schemaBaseURL>
                // + "<verbose>false</verbose>
                // + "<bboxForEachCRS>false</bboxForEachCRS>
                // + "<watermark>
                // + " <enabled>false</enabled>
                // + " <position>BOT_RIGHT</position>
                // + " <transparency>0</transparency>
                // + "</watermark>
                // + "<interpolation>Nearest</interpolation>
                // +
                // "<getFeatureInfoMimeTypeCheckingEnabled>false</getFeatureInfoMimeTypeCheckingEnabled>
                // +
                // "<getMapMimeTypeCheckingEnabled>false</getMapMimeTypeCheckingEnabled>
                // + "<maxBuffer>0</maxBuffer>
                // + "<maxRequestMemory>0</maxRequestMemory>
                // + "<maxRenderingTime>0</maxRenderingTime>
                // + "<maxRenderingErrors>0</maxRenderingErrors>
                + "</wms>";

        // This url is probably only allowed to change workspace name
        // url = geoserverRestUrl+"/workspaces/"+workspaceName

        String url = this.geoserverRestUrl + "/services/wms/workspaces/" + workspaceName + "/settings";

        URLResponse rr = HttpURLClient.sendRequest(
                url,
                null,
                this.headers,
                body,
                HttpMethod.PUT,
                null,
                this.geoserverAdmin,
                this.geoserverPass,
                false,
                false);

        if (rr.getStatusCode() < 200 || rr.getStatusCode() > 201) {
            throw new InternalError(
                    "Modification of workspace " + workspaceName + " failed with status " + rr.getStatusCode() + ".");
        }

    }

    public String getWorkspaceServiceSettings(
            String workspaceName,
            String service) throws Exception {

        String url = this.geoserverRestUrl + "/services/" + service + "/workspaces/" + workspaceName + "/settings";

        URLResponse rr = HttpURLClient.sendRequest(
                url,
                null,
                this.headers,
                null,
                HttpMethod.GET,
                null,
                this.geoserverAdmin,
                this.geoserverPass,
                false,
                false);

        if (rr.getStatusCode() < 200 || rr.getStatusCode() > 201) {
            throw new InternalError("Retrieval of workspace " + workspaceName + " settings failed with status "
                    + rr.getStatusCode() + ".");
        }

        return rr.getResponseBody();

    }

    /**
     * Change transaction operation enabled in wfs settings.
     * Can be
     * BASIC := read access,
     * TRANSACTIONAL := write access,
     * COMPLETE := read+write access
     *
     * Workspace must be enabled to do this!
     *
     * @param transactional
     */
    public void setTransactional(boolean transactional, String workspace) throws IOException {

        String body = null;

        if (transactional) {
            body = "<wfs><serviceLevel>COMPLETE</serviceLevel></wfs>";
        } else {
            body = "<wfs><serviceLevel>BASIC</serviceLevel></wfs>";
        }

        // if(transactional){
        // body = "<workspace><name>COMPLETE</name></workspace>";
        // }else{
        // body = "<wfs><serviceLevel>BASIC</serviceLevel></wfs>";
        // }

        URLResponse rr = null;
        try {
            rr = HttpURLClient.sendRequest(
                    this.geoserverRestUrl + "/services/wfs/workspaces/" + workspace + "/settings",
                    null,
                    this.headers,
                    body,
                    HttpMethod.POST,
                    null,
                    this.geoserverAdmin,
                    this.geoserverPass,
                    false,
                    false);

        } catch (Exception e) {
            e.printStackTrace();
            throw new IOException(e);
        }
        if (rr.getStatusCode() < 200 || rr.getStatusCode() > 201) {
            throw new IOException("Enabling/ Disabling of transaction support for workspace " + workspace
                    + " failed with status " + rr.getStatusCode() + ".");
        }
    }

    /**
     * Returns all datasource names, that exist on this workspace
     *
     * @param geoserverRestUrl
     * @param workspace
     * @return
     * @throws Exception
     */
    public List<String> getDatasources(String workspace) throws IOException {

        String url = "/workspaces/" + workspace + "/datastores.json";

        URLResponse rr = HttpURLClient.sendRequest(
                this.geoserverRestUrl + url,
                null,
                null,
                null,
                HttpMethod.GET,
                null,
                this.geoserverAdmin,
                this.geoserverPass,
                false,
                false);

        ArrayList<String> datasources = new ArrayList<String>();

        try {
            DataStoresList sl = JsonConverter.fromJson(rr.getResponseBody(), DataStoresList.class);

            // ArrayList<String> datasources = new ArrayList<String>();

            for (DataStores f : sl.getDataStores()) {
                for (DataStore ds : f.getDataStore()) {
                    System.out.println(ds.getName());
                    datasources.add(ds.getName());
                }
            }

        } catch (JsonMappingException jme) {
            // no datasources exist or malformed?
            // return null;
        }

        return datasources;
    }

    /**
     * Gets the first available datasource on this workspace, that is of type
     * datasourceType.
     *
     * @param workspace
     * @param datasourceType
     * @return
     * @throws Exception
     */
    public String getDatasource(String workspace, GeoserverDatasourceType datasourceType) throws IOException {

        Serializer serializer = new Persister();
        List<String> datasourceNames = this.getDatasources(workspace);

        for (String ds : datasourceNames) {

            String datasourceXML = this.getDatasourceXML(workspace, ds);
            // use strict=false to ignore unused elements
            // according to
            // https://stackoverflow.com/questions/4740934/how-to-ignore-unused-xml-elements-while-deserializing-a-document
            DataStoreXML dataStore;
            try {
                dataStore = serializer.read(DataStoreXML.class, datasourceXML, false);
            } catch (Exception e) {
                e.printStackTrace();
                throw new IOException(e);
            }

            String datasourceTypePgJNDI = datasourceType.toString();
            if (datasourceType == GeoserverDatasourceType.PostGISJNDI) {
                datasourceTypePgJNDI = "PostGIS (JNDI)";
            }

            if (dataStore.getType().equals(datasourceTypePgJNDI)
            // && dataStore.isEnabled() //TODO: should we use enabled
            // datasources only??
            ) {
                return dataStore.getName();
            }
        }

        return null;
    }

    /**
     * Get vector datasources that exist in the given workspace.
     *
     * @param workspace
     * @param datasource
     * @throws Exception
     */
    public String getDatasourceXML(String workspace, String datasource) throws IOException {

        if (datasource == null) {
            throw new IOException("Missing value datasource.");
        }

        URLResponse rr = HttpURLClient.sendRequest(
                this.geoserverRestUrl + "/workspaces/" + workspace + "/datastores/" + datasource + ".xml",
                null,
                null,
                null,
                HttpMethod.GET,
                null,
                this.geoserverAdmin,
                this.geoserverPass,
                false,
                false);

        String datasourcesXML = rr.getResponseBody();

        return datasourcesXML;
    }

}
