package de.narimo.georepo.server.layer;

import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;
import java.util.List;
import java.util.Map;

import org.geotools.feature.DefaultFeatureCollection;
import org.geotools.feature.FeatureCollection;
import org.geotools.feature.FeatureIterator;
import org.geotools.feature.simple.SimpleFeatureBuilder;
import org.geotools.feature.simple.SimpleFeatureTypeBuilder;
import org.geotools.geojson.feature.FeatureJSON;
import org.geotools.geojson.geom.GeometryJSON;
import org.geotools.referencing.crs.DefaultGeographicCRS;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.Point;
import org.opengis.feature.simple.SimpleFeature;
import org.opengis.feature.simple.SimpleFeatureType;


public class FeatureBuilder {

    final SimpleFeatureType featureType;

    public FeatureBuilder(Map<String, Class> propertyTypes) {
        this.featureType = createPointFeatureType(propertyTypes);
    }

    private static SimpleFeatureType createPointFeatureType(Map<String, Class> propertyTypes) {
        SimpleFeatureTypeBuilder builder = new SimpleFeatureTypeBuilder();
        builder.setName("geometry");
        builder.setCRS(DefaultGeographicCRS.WGS84);
        builder.setDefaultGeometry("geometry");

        builder.add("geometry", Point.class);
        for (String key : propertyTypes.keySet()) {
            builder.length(255).add(key, propertyTypes.get(key));
        }
        return builder.buildFeatureType();
    }

    public SimpleFeature createSimpleFeature(
            Geometry geometry,
            Map<String, Object> featureProperties) {

        SimpleFeatureBuilder featureBuilder = new SimpleFeatureBuilder(this.featureType);
        if (geometry != null) {
            featureBuilder.add(geometry);
        }

        SimpleFeature simpleFeature = featureBuilder.buildFeature(null);

        for (String key : featureProperties.keySet()) {
            simpleFeature.setAttribute(key, featureProperties.get(key));
        }
        return simpleFeature;
    }

    public FeatureCollection createFeatureCollection(
            List<SimpleFeature> simpleFeatures) {

        FeatureCollection featureCollection = new DefaultFeatureCollection();

        for (SimpleFeature simpleFeature : simpleFeatures) {
            ((DefaultFeatureCollection) featureCollection).add(simpleFeature);
        }
        return featureCollection;
    }

    /**
     * Writes a feature collection to json.
     * 
     * @param featureCollection
     * @return
     * @throws IOException
     */
    public static String featureCollectionToJson(FeatureCollection featureCollection) throws IOException {
        
        Writer output = new StringWriter();
        GeometryJSON gjson = new GeometryJSON(8);
        FeatureJSON fjson = new FeatureJSON(gjson);
        fjson.writeFeatureCollection(featureCollection, output);

        return output.toString();
    }
}
