package de.narimo.georepo.server.layer;

import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;
import java.util.List;

import org.geotools.feature.FeatureCollection;
import org.geotools.geojson.feature.FeatureJSON;
import org.geotools.geojson.geom.GeometryJSON;
import org.opengis.feature.simple.SimpleFeature;

import de.narimo.georepo.server.GeorepoDatasource;
import de.narimo.georepo.server.repository.DatasetRepository;
import de.narimo.georepo.server.repository.DifftableRepository;

public class FeatureLayer {

    public static String getFeatures(int userId, String dataWorkspace, String featureType) throws IOException {
        int layerId = DatasetRepository.getLayerId(dataWorkspace, featureType);
        String tableName = FeatureLayer.getDataTableName(layerId);

        FeatureCollection featureCollection = DatasetRepository.getDatasets(tableName, userId);

        if (featureCollection.isEmpty()) {
            return "{\"type\": \"FeatureCollection\", \"features\": []}";
        }
        return FeatureBuilder.featureCollectionToJson(featureCollection);
    }

    public static void addNewFeatures(long userId, String dataWorkspace, String featureType, List<SimpleFeature> features)
            throws IOException {

        int layerId = DatasetRepository.getLayerId(dataWorkspace, featureType);
        String diffTableName = FeatureLayer.getDiffTableName(layerId);
        String authTableName = FeatureLayer.getAuthTableName(layerId);
        
        DatasetRepository.insertDatasets(diffTableName, authTableName, features, userId);
    }

    public static void addModifiedFeatures(int userId, String dataWorkspace, String featureType,
            List<SimpleFeature> features) throws IOException {

        int layerId = DatasetRepository.getLayerId(dataWorkspace, featureType);
        String diffTableName = FeatureLayer.getDiffTableName(layerId);
        String authTableName = FeatureLayer.getAuthTableName(layerId);

        DatasetRepository.updateDatasets(diffTableName, authTableName, features, userId);
    }

    public static void addDeletedFeatures(int userId, String dataWorkspace, String featureType,
            List<SimpleFeature> features) throws IOException {

        int dataLayerId = DatasetRepository.getLayerId(dataWorkspace, featureType);
        String diffTableName = FeatureLayer.getDiffTableName(dataLayerId);
        String authTableName = FeatureLayer.getAuthTableName(dataLayerId);
        DatasetRepository.deleteDatasets(diffTableName, authTableName, features, userId);
    }

    public static void acceptFeatureChange(int grpfid, String dataWorkspace, String dataFeatureType) throws IOException {

        int dataLayerId = DatasetRepository.getLayerId(dataWorkspace, dataFeatureType);
        String dataTable = FeatureLayer.getDataTableName(dataLayerId);
        String diffTable = FeatureLayer.getDiffTableName(dataLayerId);
        String authTable = FeatureLayer.getAuthTableName(dataLayerId);

        DatasetRepository.acceptDataset(diffTable, dataTable, authTable, grpfid);
    }

    public static void declineFeatureChange(int grpfid, String dataWorkspace, String featureType) throws IOException {
        int dataLayerId = DatasetRepository.getLayerId(dataWorkspace, featureType);
        String diffTable = FeatureLayer.getDiffTableName(dataLayerId);
        String authTable = FeatureLayer.getAuthTableName(dataLayerId);

        DatasetRepository.declineDataset(diffTable, authTable, grpfid);
    }

    public static String getModifiedFeatures(int userId, String dataWorkspace, String featureType, boolean asAdmin)
            throws IOException {
        int layerId = DatasetRepository.getLayerId(dataWorkspace, featureType);
        String diffTableName = FeatureLayer.getDiffTableName(layerId);
        String authTableName = FeatureLayer.getAuthTableName(layerId);

        FeatureCollection featureCollection = DatasetRepository.getModifiedDatasets(diffTableName, authTableName, userId,
                asAdmin);

        if (featureCollection.isEmpty()) {
            return "{\"type\": \"FeatureCollection\", \"features\": []}";
        }
        return FeatureBuilder.featureCollectionToJson(featureCollection);
    }

    public static String getDataTableName(int layerId) {
        return GeorepoDatasource.georepoDatasourcePrefix + "" + layerId;
    }

    public static String getDiffTableName(int layerId) {
        return GeorepoDatasource.georepoDatasourcePrefix + "" + layerId + DifftableRepository.diffTableSuffix;
    }

    public static String getAuthTableName(int layerId) {
        return GeorepoDatasource.georepoDatasourcePrefix + "" + layerId + DatasetRepository.authTableSuffix;
    }
}
