package de.narimo.georepo.server.api;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

import javax.servlet.ServletContext;
import javax.ws.rs.DELETE;
import javax.ws.rs.ForbiddenException;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.SecurityContext;
import javax.ws.rs.ext.Provider;

import org.geotools.feature.FeatureIterator;
import org.geotools.geojson.feature.FeatureJSON;
import org.opengis.feature.simple.SimpleFeature;

import de.narimo.commons.dto.geometa.User;
import de.narimo.geocore.ws.repository.UserRepository;
import de.narimo.georepo.server.layer.FeatureLayer;
import de.narimo.georepo.server.notification.Notifier;
import de.narimo.georepo.server.repository.DifftableRepository;
import io.swagger.annotations.Api;

@Provider
@Path("/workspaces/{workspace}/layers/{layername}/features")
@Api(value = "FeatureController")
public class FeatureController {

    @GET
    @Produces(MediaType.APPLICATION_JSON)
    @Path("/")
    public static Response getFeatures(
            // should be determined from layerid
            @PathParam("workspace") String dataWorkspace,
            // should be determined from layerid
            @PathParam("layername") String dataLayerName, @Context SecurityContext sec,
            @Context ServletContext ctx) {

        try {

            // Features may not be retrieved from a diff table using this
            // endpoint
            // That is, double check if the requested workspace is a diff
            // workspace, then disallow!
            boolean isDiffWorkspace = isDiffWorkspace(dataWorkspace, dataLayerName);
            if (isDiffWorkspace) {
                System.out.println("diffworkspace!! it is");
                throw new ForbiddenException();
            }

            User user = UserRepository.getUser(sec.getUserPrincipal().getName()).get();
            String jsonFeatureCollection = FeatureLayer.getFeatures(user.getId(), dataWorkspace, dataLayerName);

            if (jsonFeatureCollection == null) {
                throw new InternalError("Could not get feature changes.");
            }

            return Response.ok().entity(jsonFeatureCollection).build();
        } catch (IOException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        } catch (ForbiddenException e) {
            e.printStackTrace();
            return Response.status(Status.FORBIDDEN).build();
        } catch (InternalError | RuntimeException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }

    @POST
    @Path("/")
    public static Response addFeatures(
            // should be determined from layerid
            @PathParam("workspace") String dataWorkspace,
            // should be determined from layerid
            @PathParam("layername") String dataLayerName,
            InputStream body,
            @Context SecurityContext sec,
            @Context ServletContext ctx) {

        try {
            FeatureJSON io = new FeatureJSON();
            List<SimpleFeature> features = new ArrayList<>();

            FeatureIterator<SimpleFeature> fi = io.streamFeatureCollection(body);
            while (fi.hasNext()) {
                features.add(fi.next());
            }

            if (features.size() < 1) {
                throw new IOException("No features to add.");
            }

            String diffWorkspace = DifftableRepository.getDiffWorkspace(dataWorkspace, dataLayerName);
            if (diffWorkspace == null) {
                throw new InternalError("No diff workspace defined.");
            }

            String diffLayerName = DifftableRepository.getDiffLayername(dataWorkspace, diffWorkspace, dataLayerName);
            if (diffLayerName == null) {
                throw new InternalError("No diff layer defined.");
            }

            User user = UserRepository.getUser(sec.getUserPrincipal().getName()).get();
            FeatureLayer.addNewFeatures(user.getId(), diffWorkspace, diffLayerName, features);

            boolean notifyAdmin = true;
            if (notifyAdmin) {
                Notifier.notifyAdmins(ctx, dataWorkspace, dataLayerName);
            }
        } catch (IOException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        } catch (InternalError | RuntimeException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
        return Response.ok().build();
    }

    @PUT
    @Path("/")
    public static Response modifyFeatures(
            // should be determined from layerid
            @PathParam("workspace") String dataWorkspace,
            // should be determined from layerid
            @PathParam("layername") String dataLayerName,
            InputStream body,
            @Context SecurityContext sec,
            @Context ServletContext ctx) {

        try {
            FeatureJSON io = new FeatureJSON();
            List<SimpleFeature> features = new ArrayList<>();

            FeatureIterator<SimpleFeature> fi = io.streamFeatureCollection(body);
            while (fi.hasNext()) {
                features.add(fi.next());
            }

            if (features.size() < 1) {
                throw new IOException("No features to add.");
            }

            String diffWorkspace = DifftableRepository.getDiffWorkspace(dataWorkspace, dataLayerName);
            if (diffWorkspace == null) {
                throw new InternalError("No diff workspace defined.");
            }

            String diffLayerName = DifftableRepository.getDiffLayername(dataWorkspace, diffWorkspace, dataLayerName);
            if (diffLayerName == null) {
                throw new InternalError("No diff layer defined.");
            }

            String username = sec.getUserPrincipal().getName();
            int userId = UserRepository.getUserId(username);

            FeatureLayer.addModifiedFeatures(userId, diffWorkspace, diffLayerName, features);

            boolean notifyAdmin = true;
            if (notifyAdmin) {
                Notifier.notifyAdmins(ctx, dataWorkspace, dataLayerName);
            }
        } catch (IOException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        } catch (InternalError | RuntimeException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
        return Response.ok().build();
    }

    @DELETE
    @Path("/")
    public static Response deleteFeatures(
            // should be determined from layerid
            @PathParam("workspace") String dataWorkspace,
            // should be determined from layerid
            @PathParam("layername") String dataLayerName,
            InputStream body,
            @Context SecurityContext sec,
            @Context ServletContext ctx) {

        try {
            FeatureJSON io = new FeatureJSON();
            List<SimpleFeature> features = new ArrayList<>();

            FeatureIterator<SimpleFeature> fi = io.streamFeatureCollection(body);
            while (fi.hasNext()) {
                features.add(fi.next());
            }

            if (features.size() < 1) {
                throw new IOException("No features to delete.");
            }

            String diffWorkspace = DifftableRepository.getDiffWorkspace(dataWorkspace, dataLayerName);
            if (diffWorkspace == null) {
                throw new InternalError("No diff workspace defined.");
            }

            String diffLayerName = DifftableRepository.getDiffLayername(dataWorkspace, diffWorkspace, dataLayerName);
            if (diffLayerName == null) {
                throw new InternalError("No diff layer defined.");
            }

            String username = sec.getUserPrincipal().getName();
            int userId = UserRepository.getUserId(username);

            FeatureLayer.addDeletedFeatures(userId, diffWorkspace, diffLayerName, features);

            boolean notifyAdmin = true;
            if (notifyAdmin) {
                Notifier.notifyAdmins(ctx, dataWorkspace, dataLayerName);
            }
        } catch (IOException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        } catch (InternalError | RuntimeException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
        return Response.ok().build();
    }

    /**
     * Are data workspace and diff workspace the same?
     *
     * @param dataWorkspace
     * @param dataLayerName
     * @return
     */
    private static boolean isDiffWorkspace(String dataWorkspace, String dataLayerName) {
        String diffWorkspace = DifftableRepository.getDiffWorkspace(dataWorkspace, dataLayerName);
        String diffLayerName = DifftableRepository.getDiffLayername(dataWorkspace, diffWorkspace, dataLayerName);
        return dataWorkspace.equals(diffWorkspace) || dataLayerName.equals(diffLayerName);
    }
}
