package de.narimo.georepo.server.api;

import java.io.IOException;

import javax.ws.rs.DELETE;
import javax.ws.rs.ForbiddenException;
import javax.ws.rs.GET;
import javax.ws.rs.NotFoundException;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.SecurityContext;
import javax.ws.rs.ext.Provider;

import de.narimo.geocore.ws.repository.UserRepository;
import de.narimo.georepo.server.layer.FeatureLayer;
import de.narimo.georepo.server.repository.DifftableRepository;
import de.narimo.georepo.server.tools.AdminTools;
import io.swagger.annotations.Api;

@Provider
@Path("/workspaces/{workspace}/layers/{layername}/features/changes")
@Api(value = "FeatureModificationController")
public class FeatureModificationController {

    /**
     * Returns all feature on the given layer that has been submitted as changes
     * by the user.
     * No changes are returned if they have already been accepted and merged to
     * the original dataset.
     *
     * @param dataWorkspace
     * @param dataLayerName
     * @param body
     * @param sec
     * @return
     */
    @GET
    @Produces(MediaType.APPLICATION_JSON)
    @Path("/")
    public static Response getFeatureModifications(
            // should be determined from layerid
            @PathParam("workspace") String dataWorkspace,
            // should be determined from layerid
            @PathParam("layername") String dataLayerName,
            @QueryParam("admin") String requestAsAdmin,
            @Context SecurityContext sec) {

        try {
            String diffWorkspace = DifftableRepository.getDiffWorkspace(dataWorkspace, dataLayerName);
            if (diffWorkspace == null) {
                throw new InternalError("No diff workspace defined.");
            }

            String diffLayerName = DifftableRepository.getDiffLayername(dataWorkspace, diffWorkspace, dataLayerName);
            if (diffLayerName == null) {
                throw new InternalError("No diff layer defined.");
            }

            String username = sec.getUserPrincipal().getName();
            int userId = UserRepository.getUserId(username);

            boolean asAdmin = false;
            if (requestAsAdmin != null && requestAsAdmin.equals("true")) {
                AdminTools.checkAdminPermission(userId, diffWorkspace);
                asAdmin = true;
                System.out.println("As admin? - " + asAdmin);
            }

            String jsonFeatureCollection = FeatureLayer.getModifiedFeatures(userId,
                    diffWorkspace, diffLayerName, asAdmin);
            if (jsonFeatureCollection == null) {
                throw new InternalError("Could not get feature changes.");
            }
            return Response.ok().entity(jsonFeatureCollection).build();
        } catch (IOException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        } catch (ForbiddenException e) {
            e.printStackTrace();
            return Response.status(Status.FORBIDDEN).build();
        } catch (InternalError | RuntimeException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }

    @PUT
    @Path("/{grpfid}/accept")
    public static Response acceptFeatureChangeset(
            @PathParam("workspace") String dataWorkspace,
            @PathParam("layername") String dataLayerName,
            @PathParam("grpfid") String grpfidIn,
            @Context SecurityContext sec) {

        try {

            Integer grpfidI = Integer.valueOf(grpfidIn);
            if (grpfidI == null) {
                throw new IOException("Invalid feature id: null.");
            }
            int grpfid = grpfidI.intValue();

            String username = sec.getUserPrincipal().getName();
            int userId = UserRepository.getUserId(username);

            AdminTools.checkAdminPermission(userId, dataWorkspace);

            // TODO: copy changeset to original data table
            String diffWorkspace = DifftableRepository.getDiffWorkspace(dataWorkspace, dataLayerName);
            if (diffWorkspace == null) {
                throw new InternalError("No diff workspace defined.");
            }

            String diffLayerName = DifftableRepository.getDiffLayername(dataWorkspace, diffWorkspace, dataLayerName);
            if (diffLayerName == null) {
                throw new InternalError("No diff layer defined.");
            }

            FeatureLayer.acceptFeatureChange(grpfid, dataWorkspace, dataLayerName, diffWorkspace, diffLayerName);
            return Response.ok().build();

        } catch (NumberFormatException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).entity("Invalid feature id.").build();
        } catch (NotFoundException e) {
            e.printStackTrace();
            return Response.status(Status.NOT_FOUND).build();
        } catch (IOException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        } catch (ForbiddenException e) {
            e.printStackTrace();
            return Response.status(Status.FORBIDDEN).build();
        } catch (InternalError | RuntimeException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }

    @DELETE
    @Path("/{grpfid}/decline")
    public static Response declineFeatureChangeset(
            @PathParam("workspace") String dataWorkspace,
            @PathParam("layername") String dataLayerName,
            @PathParam("grpfid") String grpfidIn,
            @Context SecurityContext sec) {

        try {

            Integer grpfidI = Integer.valueOf(grpfidIn);
            if (grpfidI == null) {
                throw new IOException("Invalid feature id: null.");
            }
            int grpfid = grpfidI.intValue();

            String username = sec.getUserPrincipal().getName();
            int userId = UserRepository.getUserId(username);

            AdminTools.checkAdminPermission(userId, dataWorkspace);

            // TODO: copy changeset to original data table
            String diffWorkspace = DifftableRepository.getDiffWorkspace(dataWorkspace, dataLayerName);
            if (diffWorkspace == null) {
                throw new InternalError("No diff workspace defined.");
            }

            String diffLayerName = DifftableRepository.getDiffLayername(dataWorkspace, diffWorkspace, dataLayerName);
            if (diffLayerName == null) {
                throw new InternalError("No diff layer defined.");
            }

            FeatureLayer.declineFeatureChange(grpfid, diffWorkspace, diffLayerName);
            return Response.ok().build();

        } catch (NumberFormatException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).entity("Invalid feature id.").build();
        } catch (NotFoundException e) {
            e.printStackTrace();
            return Response.status(Status.NOT_FOUND).build();
        } catch (IOException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        } catch (ForbiddenException e) {
            e.printStackTrace();
            return Response.status(Status.FORBIDDEN).build();
        } catch (InternalError | RuntimeException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }
}
