package de.narimo.georepo.server.layer;

import java.io.IOException;
import java.util.List;

import org.geotools.feature.FeatureCollection;
import org.geotools.geojson.feature.FeatureJSON;
import org.opengis.feature.simple.SimpleFeature;

import de.narimo.georepo.server.GeorepoDatasource;
import de.narimo.georepo.server.repository.DatasetRepository;
import de.narimo.georepo.server.repository.DifftableRepository;

public class FeatureLayer {

    public static String getFeatures(int userId, String workspace, String featureType) throws IOException {
        int layerId = DatasetRepository.getLayerId(workspace, featureType);
        String tableName = FeatureLayer.getDataTableName(layerId);

        FeatureCollection featureCollection = DatasetRepository.getDatasets(tableName, userId);
        FeatureJSON io = new FeatureJSON();

        if (featureCollection.isEmpty()) {
            return "{\"type\": \"FeatureCollection\", \"features\": []}";
        }
        return io.toString(featureCollection);
    }

    public static void addNewFeatures(long userId, String workspace, String featureType, List<SimpleFeature> features)
            throws IOException {

        int layerId = DatasetRepository.getLayerId(workspace, featureType);
        String tableName = FeatureLayer.getDataTableName(layerId);

        DatasetRepository.insertDatasets(tableName, features, userId);
    }

    public static void addModifiedFeatures(int userId, String workspace, String featureType,
            List<SimpleFeature> features) throws IOException {

        int layerId = DatasetRepository.getLayerId(workspace, featureType);
        String tableName = FeatureLayer.getDataTableName(layerId);

        DatasetRepository.updateDatasets(tableName, features, userId);
    }

    public static void addDeletedFeatures(int userId, String workspace, String featureType,
            List<SimpleFeature> features) throws IOException {

        int layerId = DatasetRepository.getLayerId(workspace, featureType);
        String tableName = FeatureLayer.getDataTableName(layerId);
        DatasetRepository.deleteDatasets(tableName, features, userId);
    }

    public static void acceptFeatureChange(int grpfid, String dataWorkspace, String dataFeatureType,
            String diffWorkspace, String diffFeatureType) {

        int dataLayerId = DatasetRepository.getLayerId(dataWorkspace, dataFeatureType);
        String dataTable = FeatureLayer.getDataTableName(dataLayerId);

        int diffLayerId = DatasetRepository.getLayerId(diffWorkspace, diffFeatureType);
        String diffTable = FeatureLayer.getDataTableName(diffLayerId);

        DatasetRepository.acceptDataset(diffTable, dataTable, grpfid);
    }

    public static void declineFeatureChange(int grpfid, String diffWorkspace, String diffFeatureType) {
        int diffLayerId = DatasetRepository.getLayerId(diffWorkspace, diffFeatureType);
        String diffTable = FeatureLayer.getDataTableName(diffLayerId);
        DatasetRepository.declineDataset(diffTable, grpfid);
    }

    public static String getModifiedFeatures(int userId, String workspace, String featureType, boolean asAdmin)
            throws IOException {
        int layerId = DatasetRepository.getLayerId(workspace, featureType);
        String tableName = FeatureLayer.getDataTableName(layerId);

        FeatureCollection featureCollection = DatasetRepository.getModifiedDatasets(tableName, userId, asAdmin);
        FeatureJSON io = new FeatureJSON();

        if (featureCollection.isEmpty()) {
            return "{\"type\": \"FeatureCollection\", \"features\": []}";
        }
        return io.toString(featureCollection);
    }

    public static String getDataTableName(int layerId) {
        return GeorepoDatasource.georepoDatasourcePrefix + "" + layerId;
    }

    public static String getDiffTableName(int layerId) {
        return GeorepoDatasource.georepoDatasourcePrefix + "" + layerId + DifftableRepository.diffTableSuffix;
    }

    public static String getAuthTableName(int layerId) {
        return GeorepoDatasource.georepoDatasourcePrefix + "" + layerId + DatasetRepository.authTableSuffix;
    }
}
