package de.narimo.georepo.server.api;

import java.io.IOException;
import java.sql.SQLException;

import javax.servlet.ServletContext;
import javax.ws.rs.ForbiddenException;
import javax.ws.rs.NotFoundException;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.SecurityContext;
import javax.ws.rs.ext.Provider;

import de.narimo.commons.dto.geometa.User;
import de.narimo.geocore.ws.repository.UserRepository;
import de.narimo.georepo.server.repository.DatasetRepository;
import de.narimo.georepo.server.repository.DifftableRepository;
import de.narimo.georepo.server.tools.AdminTools;
import de.narimo.georepo.server.tools.TableTools;
import io.swagger.annotations.Api;

@Provider
@Path("/workspaces/{workspace}/layers/")
@Api(value = "DiffLayerController")
public class DiffLayerController {

    /**
     * Creates a new diff and auth table together with a diff workspace
     * and inserts to table difftables.
     *
     * @param sec
     * @param ctx
     * @param dataWorkspace
     * @param dataLayerName
     * @param userName
     * @return
     * @throws IOException
     */
    @PUT
    @Path("/{layername}/managed")
    public static Response createManageableDiffLayer(
            @Context SecurityContext sec,
            @Context ServletContext ctx,
            @PathParam("workspace") String dataWorkspace,
            @PathParam("layername") String dataLayerName)
            throws IOException {

        try {

            String username = sec.getUserPrincipal().getName();
            User user = UserRepository.getUser(username).get();

            AdminTools.checkAdminPermission(user.getId(), dataWorkspace);

            int layerId = DatasetRepository.getLayerId(dataWorkspace, dataLayerName);
            String dataTable = TableTools.getDataTableName(layerId);
            String diffTable = TableTools.getDiffTableName(layerId);
            String diffAuthTable = TableTools.getDiffAuthTableName(layerId);

            boolean diffTableExists = DifftableRepository.checkDiffTableExists(diffTable);
            if (!diffTableExists) {

                DifftableRepository.createDiffTable(dataTable, diffTable);
                DifftableRepository.createDiffAuthTable(diffAuthTable);

                String diffWorkspace = null;

                try {
                    diffWorkspace = WorkspacesController.createNewWorkspace(ctx, user);
                } catch (IOException e) {
                    if (!e.getMessage().contains("Enabling/ Disabling of transaction support")) {
                        throw e;
                    } else {
                        // This is expected behaviour since the workspace
                        // needs to be activated for WMS/ WFS access.
                        // We don't need it in this case, so we ignore it.
                    }
                }

                // Do NOT add to table datasets!
                // Only datasets imported with import will be placed there to
                // determine the data table id

                // add diff table to a workspace, i.e.
                // add diff table to difftables table
                DifftableRepository.addDiffLayer(TableTools.getDiffLayerName(dataLayerName),
                        diffWorkspace, dataLayerName, dataWorkspace);

                WorkspaceController.setWorkspacePermission(user, user, diffWorkspace, "rw");

            } else {
                System.out.println("Diff table " + diffTable + " exists. Assuming, a manageable layer for "
                        + dataLayerName + " already exists.");
            }

        } catch (ForbiddenException e) {
            e.printStackTrace();
            return Response.status(Status.FORBIDDEN).build();
        } catch (NotFoundException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        } catch (SQLException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        } catch (Exception e) {
            e.printStackTrace();
            Response.status(Status.BAD_REQUEST).entity("Could not create manageable layer. ").build();
        }

        return Response.ok().build();
    }
}
