package de.narimo.georepo.server.notification;

import java.util.List;

import javax.servlet.ServletContext;

import de.narimo.commons.dto.geometa.User;
import de.narimo.commons.ws.mail.Mailer;
import de.narimo.geocore.ws.registration.PasswordForgottenDetails;
import de.narimo.geocore.ws.registration.UserRegistrationDetails;
import de.narimo.geocore.ws.repository.UserRepository;
import de.narimo.georepo.server.repository.WorkspaceRepository;

public class Notifier {

    /**
     * Send notification to workspace admins that POI changes have been
     * submitted
     */
    public static void notifyAdmins(ServletContext ctx, String workspace, String dataLayerName) {

        Thread thread = new Thread("notifyAdmins") {
            @Override
            public void run() {
                List<Integer> adminIds = WorkspaceRepository.getWorkspaceAdminIds(workspace);
                List<User> admins = UserRepository.getUsersById(adminIds);
                for (User admin : admins) {
                    Notifier.sendPoiUpdateAdminNotification(ctx, dataLayerName, admin.getEmail());
                }
            }
        };
        thread.start();
    }

    /**
     * Send a notification that POI changes have been submitted.
     *
     * @param ctx
     * @param dataLayerName
     * @param recipient
     */
    private static void sendPoiUpdateAdminNotification(ServletContext ctx, String dataLayerName, String recipient) {
        Mailer mailer = new Mailer();

        String mailerUser = ctx.getInitParameter("info_mail_user");
        String mailerPass = ctx.getInitParameter("info_mail_pw");

        String sender = "info@georepo.com";

        String subject = "Your georepo dataset has been updated";
        String msgBody = "Hi there!\n\nFor your data layer " + dataLayerName
                + " an update has been suggested by a user. "
                + "Please review the change to either accept or decline it."
                + "\n\n"
                + "This mail was automatically created. Please do not reply to it."
                + "\n\n"
                + "Your georepo team";
        mailer.sendMail(sender, recipient, subject, msgBody, mailerUser, mailerPass);
    }

    /**
     * Send notification when a new user has registered.
     */
    public static void notifyRegistrationConfirmation(ServletContext ctx,
            UserRegistrationDetails registrationDetails,
            String registrationLink) {

        Thread thread = new Thread("notifyRegistration") {
            @Override
            public void run() {
                Mailer mailer = new Mailer();

                String mailerUser = ctx.getInitParameter("info_mail_user");
                String mailerPass = ctx.getInitParameter("info_mail_pw");

                String sender = "info@georepo.com";

                String defaultApplication = "georepo";
                String registeredApplication = registrationDetails.getClientApp() != null
                        && !registrationDetails.getClientApp().isEmpty() ? registrationDetails.getClientApp()
                                : defaultApplication;

                String subject = "Your " + registeredApplication + " registration";
                String msgBody = "Hi there!\n\n"
                        + "A new account has been registered with your email."
                        + " Please click the following link to confirm your registration."
                        + "\n\n"
                        + registrationLink
                        + "\n\n"
                        + "If you did not register, you can just ignore this email."
                        + "\n\n"
                        + "Your " + defaultApplication + " team";
                mailer.sendMail(sender, registrationDetails.getEmail(), subject, msgBody, mailerUser, mailerPass);
            }
        };
        thread.start();
    }

    /**
     * Notify narimo that a registration has been requested.
     *
     * @param ctx
     * @param registrationDetails
     * @param registrationLink
     */
    public static void notifyNarimoAboutRegistration(ServletContext ctx,
            UserRegistrationDetails registrationDetails,
            String registrationLink) {

        Thread thread = new Thread("notifyNarimoAboutRegistration") {
            @Override
            public void run() {
                Mailer mailer = new Mailer();

                String mailerUser = ctx.getInitParameter("info_mail_user");
                String mailerPass = ctx.getInitParameter("info_mail_pw");

                String sender = "info@georepo.com";

                String defaultApplication = "georepo";
                String registeredApplication = registrationDetails.getClientApp() != null
                        && !registrationDetails.getClientApp().isEmpty() ? registrationDetails.getClientApp()
                                : defaultApplication;

                String subject = "A new user has requested registration for " + registeredApplication;
                String msgBody = "Hi narimo!\n\n"
                        + "FYI:"
                        + "\n\nA new account request has been issued for user " + registrationDetails.getEmail() + "."
                        + "\n\n"
                        + "Your " + defaultApplication + " team";
                mailer.sendMail(sender, "info@narimo.de", subject, msgBody, mailerUser, mailerPass);
            }
        };
        thread.start();
    }

    /**
     * Send password reset mail when a password forgotten request was submitted.
     */
    public static void notifyPasswordResetLink(ServletContext ctx,
            PasswordForgottenDetails passwordDetails,
            String resetLink) {

        Thread thread = new Thread("notifyPasswordForgotten") {
            @Override
            public void run() {
                Mailer mailer = new Mailer();

                String mailerUser = ctx.getInitParameter("info_mail_user");
                String mailerPass = ctx.getInitParameter("info_mail_pw");

                String sender = "info@georepo.com";

                String defaultApplication = "georepo";
                String registeredApplication = passwordDetails.getClientApp() != null
                        && !passwordDetails.getClientApp().isEmpty() ? passwordDetails.getClientApp()
                                : defaultApplication;

                String subject = "Your " + registeredApplication + " password reset request";
                String msgBody = "Hi there!\n\n"
                        + "A password reset request has been issued for your email. "
                        + "Please click the following link to issue a new password."
                        + "\n\n"
                        + resetLink
                        + "\n\n"
                        + "If you did not issue the password reset, you can just ignore this email."
                        + "\n\n"
                        + "Your " + defaultApplication + " team";
                // TODO: It is not completely clear how much the password reset
                // mail needs to be customized to the customer application
                // + "Your " + registeredApplication + " team";
                mailer.sendMail(sender, passwordDetails.getEmail(), subject, msgBody, mailerUser, mailerPass);
            }
        };
        thread.start();
    }

    public static void notifyPasswordResetSuccessful(ServletContext ctx, String recipientMail) {

        Thread thread = new Thread("notifyPasswordResetSuccessful") {
            @Override
            public void run() {
                Mailer mailer = new Mailer();

                String mailerUser = ctx.getInitParameter("info_mail_user");
                String mailerPass = ctx.getInitParameter("info_mail_pw");

                String sender = "info@georepo.com";

                String defaultApplication = "georepo";

                String subject = "Your " + defaultApplication + " password has been changed";
                String msgBody = "Hi there!\n\n"
                        + "Your password has been changed successfully. "
                        + "If this wasn't you, please contact customer support by replying to this mail."
                        + "\n\n"
                        + "Your " + defaultApplication + " team";
                mailer.sendMail(sender, recipientMail, subject, msgBody, mailerUser, mailerPass);
            }
        };
        thread.start();
    }

    public static void notifyAccountDeleteConfirmation(ServletContext ctx, String recipientMail, String clientApp) {

        Thread thread = new Thread("notifyAccountDeleteConfirmation") {
            @Override
            public void run() {
                Mailer mailer = new Mailer();

                String mailerUser = ctx.getInitParameter("info_mail_user");
                String mailerPass = ctx.getInitParameter("info_mail_pw");

                String sender = "info@georepo.com";

                String defaultApplication = "georepo";
                String registeredApplication = clientApp != null && !clientApp.isEmpty() ? clientApp : defaultApplication;

                String subject = "Your " + registeredApplication + " account has been deleted";
                String msgBody = "Hi there!\n\n"
                        + "We have processed your request to delete your account. But we regret that you are leaving "
                        + "and would always like to see you back at some point."
                        + "\n\nWe would be happy if you let us know what went wrong by just replying to this email to support enhancing our services."
                        + "\n\n"
                        + "Your " + registeredApplication + " team";
                mailer.sendMail(sender, recipientMail, subject, msgBody, mailerUser, mailerPass);
            }
        };
        thread.start();
    }

    public static void notifyNarimoAboutAccountRemoval(ServletContext ctx, String userAccountMail, String clientApp) {

        Thread thread = new Thread("notifyNarimoAboutAccountDeletion") {
            @Override
            public void run() {
                Mailer mailer = new Mailer();

                String mailerUser = ctx.getInitParameter("info_mail_user");
                String mailerPass = ctx.getInitParameter("info_mail_pw");

                String sender = "info@georepo.com";

                String defaultApplication = "georepo";
                String registeredApplication = clientApp != null && !clientApp.isEmpty() ? clientApp : defaultApplication;

                String subject = "An account has been deleted for " + registeredApplication + " application";
                String msgBody = "Hi narimo!\n\n" + "FYI:" + "\n\nAn account delete request has been processed for user "
                        + userAccountMail + "." + "\n\n" + "Your " + defaultApplication + " team";
                mailer.sendMail(sender, "info@narimo.de", subject, msgBody, mailerUser, mailerPass);
            }
        };
        thread.start();
    }

    public static void notifyNarimoAboutFailedAccountRemoval(ServletContext ctx, String userAccountMail, String clientApp) {

        Thread thread = new Thread("notifyNarimoAboutAccountDeletion") {
            @Override
            public void run() {
                Mailer mailer = new Mailer();

                String mailerUser = ctx.getInitParameter("info_mail_user");
                String mailerPass = ctx.getInitParameter("info_mail_pw");

                String sender = "info@georepo.com";

                String defaultApplication = "georepo";
                String registeredApplication = clientApp != null && !clientApp.isEmpty() ? clientApp : defaultApplication;

                String subject = "Possible failed account removal for " + registeredApplication + " application";
                String msgBody = "Hi narimo!\n\n" + "TO BE DEBUGGED:"
                        + "\n\nAn account delete request has been processed for user " + userAccountMail
                        + " but might have failed!\n\n"
                        + "Please check user database and server logs to confirm correct account removal!\n\n" + "Your "
                        + defaultApplication + " team";
                mailer.sendMail(sender, "info@narimo.de", subject, msgBody, mailerUser, mailerPass);
            }
        };
        thread.start();
    }
}
