package de.narimo.georepo.server.api;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

import javax.servlet.ServletContext;
import javax.ws.rs.DELETE;
import javax.ws.rs.ForbiddenException;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.SecurityContext;
import javax.ws.rs.ext.Provider;

import org.geotools.feature.FeatureIterator;
import org.geotools.geojson.feature.FeatureJSON;
import org.opengis.feature.simple.SimpleFeature;

import de.narimo.commons.dto.geometa.User;
import de.narimo.geocore.ws.repository.UserRepository;
import de.narimo.georepo.server.layer.DiffFeatureLayer;
import de.narimo.georepo.server.notification.Notifier;
import de.narimo.georepo.server.repository.DifftableRepository;
import de.narimo.georepo.server.tools.AdminTools;
import io.swagger.annotations.Api;

@Provider
@Path("/workspaces/{workspace}/layers/{layername}/features/changes")
@Api(value = "FeatureModificationController")
public class FeatureModificationController {

    /**
     * Returns all feature on the given layer that has been submitted as changes
     * by the user.
     * No changes are returned if they have already been accepted and merged to
     * the original dataset.
     *
     * @param dataWorkspace
     * @param dataLayerName
     * @param body
     * @param sec
     * @return
     */
    @GET
    @Produces(MediaType.APPLICATION_JSON)
    @Path("/")
    public static Response getFeatureModifications(
            // should be determined from layerid
            @PathParam("workspace") String diffWorkspace,
            // should be determined from layerid
            @PathParam("layername") String dataLayerName,
            @QueryParam("admin") String requestAsAdmin,
            @Context SecurityContext sec) {

        try {

            String dataWorkspace = DifftableRepository.getDataWorkspace(diffWorkspace, dataLayerName);
            if (dataWorkspace == null) {
                throw new InternalError("No data workspace defined.");
            }

            String username = sec.getUserPrincipal().getName();
            int userId = UserRepository.getUserId(username);

            boolean asAdmin = false;
            if (requestAsAdmin != null && requestAsAdmin.equals("true")) {
                AdminTools.checkAdminPermission(userId, diffWorkspace);
                asAdmin = true;
                System.out.println("As admin? - " + asAdmin);
            }

            String jsonFeatureCollection = DiffFeatureLayer.getModifiedFeatures(userId,
                    dataWorkspace, dataLayerName, asAdmin);
            if (jsonFeatureCollection == null) {
                throw new InternalError("Could not get feature changes.");
            }
            return Response.ok().entity(jsonFeatureCollection).build();
        } catch (IOException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        } catch (ForbiddenException e) {
            e.printStackTrace();
            return Response.status(Status.FORBIDDEN).build();
        } catch (InternalError | RuntimeException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }

    /**
     * This will be executed on a diff workspace since the user does not have write permission on a data workspace.
     * 
     * @param dataWorkspace
     * @param dataLayerName
     * @param body
     * @param sec
     * @param ctx
     * @return
     */
    @POST
    @Path("/")
    public static Response addFeatures(
            // should be determined from layerid
            @PathParam("workspace") String diffWorkspace,
            // should be determined from layerid
            @PathParam("layername") String dataLayerName, InputStream body, @Context SecurityContext sec,
            @Context ServletContext ctx) {

        try {
            FeatureJSON io = new FeatureJSON();
            List<SimpleFeature> features = new ArrayList<>();

            FeatureIterator<SimpleFeature> fi = io.streamFeatureCollection(body);
            while (fi.hasNext()) {
                features.add(fi.next());
            }

            if (features.size() < 1) {
                throw new IOException("No features to add.");
            }

            String dataWorkspace = DifftableRepository.getDataWorkspace(diffWorkspace, dataLayerName);
            if (dataWorkspace == null) {
                throw new InternalError("No data workspace defined.");
            }

            User user = UserRepository.getUser(sec.getUserPrincipal().getName()).get();
            DiffFeatureLayer.addNewFeatures(user.getId(), dataWorkspace, dataLayerName, features);

            boolean notifyAdmin = true;
            if (notifyAdmin) {
                Notifier.notifyAdmins(ctx, dataWorkspace, dataLayerName);
            }
        } catch (IOException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        } catch (InternalError | RuntimeException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
        return Response.ok().build();
    }

    @PUT
    @Path("/")
    public static Response modifyFeatures(
            // should be determined from layerid
            @PathParam("workspace") String diffWorkspace,
            // should be determined from layerid
            @PathParam("layername") String dataLayerName, InputStream body, @Context SecurityContext sec,
            @Context ServletContext ctx) {

        try {
            FeatureJSON io = new FeatureJSON();
            List<SimpleFeature> features = new ArrayList<>();

            FeatureIterator<SimpleFeature> fi = io.streamFeatureCollection(body);
            while (fi.hasNext()) {
                features.add(fi.next());
            }

            if (features.size() < 1) {
                throw new IOException("No features to add.");
            }

            String dataWorkspace = DifftableRepository.getDataWorkspace(diffWorkspace, dataLayerName);
            if (dataWorkspace == null) {
                throw new InternalError("No data workspace defined.");
            }

            String username = sec.getUserPrincipal().getName();
            int userId = UserRepository.getUserId(username);

            DiffFeatureLayer.addModifiedFeatures(userId, dataWorkspace, dataLayerName, features);

            boolean notifyAdmin = true;
            if (notifyAdmin) {
                Notifier.notifyAdmins(ctx, dataWorkspace, dataLayerName);
            }
        } catch (IOException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        } catch (InternalError | RuntimeException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
        return Response.ok().build();
    }

    @DELETE
    @Path("/")
    public static Response deleteFeatures(
            // should be determined from layerid
            @PathParam("workspace") String diffWorkspace,
            // should be determined from layerid
            @PathParam("layername") String dataLayerName, InputStream body, @Context SecurityContext sec,
            @Context ServletContext ctx) {

        try {
            FeatureJSON io = new FeatureJSON();
            List<SimpleFeature> features = new ArrayList<>();

            FeatureIterator<SimpleFeature> fi = io.streamFeatureCollection(body);
            while (fi.hasNext()) {
                features.add(fi.next());
            }

            if (features.size() < 1) {
                throw new IOException("No features to delete.");
            }

            String dataWorkspace = DifftableRepository.getDataWorkspace(diffWorkspace, dataLayerName);
            if (dataWorkspace == null) {
                throw new InternalError("No data workspace defined.");
            }

            String username = sec.getUserPrincipal().getName();
            int userId = UserRepository.getUserId(username);

            DiffFeatureLayer.addDeletedFeatures(userId, dataWorkspace, dataLayerName, features);

            boolean notifyAdmin = true;
            if (notifyAdmin) {
                Notifier.notifyAdmins(ctx, dataWorkspace, dataLayerName);
            }
        } catch (IOException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        } catch (InternalError | RuntimeException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
        return Response.ok().build();
    }
}
