package de.narimo.georepo.server.api;

import java.io.File;
import java.io.IOException;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.util.ArrayList;
import java.util.List;

import javax.servlet.ServletContext;
import javax.ws.rs.Consumes;
import javax.ws.rs.FormParam;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.SecurityContext;
import javax.ws.rs.ext.Provider;

import de.narimo.commons.dto.GeorepoFeatureType;
import de.narimo.commons.jdbc.JDBCConnectionJNDI;
import de.narimo.commons.json.JsonConverter;
import de.narimo.georepo.server.GeorepoDatasource;
import de.narimo.georepo.server.GeorepoFeature;
import de.narimo.georepo.server.db.PostgisInput;
import de.narimo.georepo.server.dto.FeatureType;
import de.narimo.georepo.server.geoserver.GeoserverLayer;
import de.narimo.georepo.server.geoserver.GeoserverWorkspace;
import de.narimo.georepo.server.tools.ImportTools;
import de.narimo.georepo.server.ui.ImportUI;
import io.swagger.annotations.Api;

@Provider
@Path("/workspaces/{workspace}")
@Api(value = "ImportController")
public class ImportController {

    @POST
    @Path("/upload")
    public static Response upload() {
        PostgisInput.uploadFile();
        return Response.ok().build();
    }

    @POST
    @Path("/import")
    @Consumes(MediaType.APPLICATION_FORM_URLENCODED)
    public static Response importFile(
            @Context SecurityContext sec,
            @Context ServletContext ctx,
    
            @PathParam("workspace") String workspace,
    
            @FormParam("layertitle") String layerTitle,
            @FormParam("columntypes") String optColumnTypes,

            // for local file import
            @FormParam("filename") String fileName,

            // for remote file import
            @FormParam("csvdelimiter") String usedCSVDelimiter,
            @FormParam("remotecsvurl") String remoteCSVUrl,
            @FormParam("decimalsep") String decimalSep,
            @FormParam("thousandssep") String thousandsSep,
            @FormParam("lonname") String optLongitudeName,
            @FormParam("latname") String optLatitudeName,
            @FormParam("approved") String approvedString,
            @FormParam("keepRowColumnName") String keepRowColumnName,
            @FormParam("keepRowColumnValue") String keepRowColumnValue
    ) throws Exception {

        try {
            if (fileName != null) {
                ImportTools.importCsvFile(ctx, workspace, fileName, layerTitle, optColumnTypes);
                
            }else {
                if (workspace == null) {
                    throw new IOException("Workspace must not defined.");
                }
                if (layerTitle == null) {
                    throw new IllegalArgumentException("Missing or wrong parameter layertitle.");
                }
                if (optColumnTypes == null) {
                    throw new IllegalArgumentException("Missing or wrong parameter columntypes.");
                }
                if (usedCSVDelimiter == null) {
                    throw new IllegalArgumentException("Missing or wrong parameter csvdelimiter.");
                }
                if (remoteCSVUrl == null) {
                    throw new IllegalArgumentException("Missing or wrong parameter remotecsvurl.");
                }

                boolean approved = Boolean.parseBoolean(approvedString);

                ImportTools.importRemoteFile(ctx, layerTitle, optColumnTypes, workspace, remoteCSVUrl, usedCSVDelimiter,
                        decimalSep, thousandsSep, optLongitudeName, optLatitudeName, approved, keepRowColumnName,
                        keepRowColumnValue);
                
            }
        } catch (IOException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST)
                    .entity(new File(ImportController.class.getResource("error.html").getFile()))
                    .build();
        } catch (Exception e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR)
                    .entity(new File(ImportController.class.getResource("error.html").getFile()))
                    .build();
        }
        File found = new File(ImportController.class.getResource("found.html").getFile());

        return Response.ok().entity(found).build();

    }

    @GET
    @Path("/import/ui")
    public static Response getImportUI(@PathParam("workspace") String workspace) {

        StringBuilder s = new StringBuilder();

        s.append("<html>");
        s.append("<head>");
        s.append("<meta charset=\"utf-8\"/>");
        s.append("<title>georepo importer</title>");
        s.append("<meta name=\"robots\" content=\"noindex,nofollow\">");
        s.append("</head>");
        s.append("<body>");

        s.append("<p><b>narimo systems geodata updater</b></p>");

        String importUrl = "/georepo-server/api/workspaces/" + workspace + "/import?";

        s.append(ImportUI.getLayerCreateForm(importUrl));
        s.append("</br>");

        s.append("</body>");
        s.append("</html>");

        return Response.ok().entity(s.toString()).build();
    }

    /**
     * Returns all feature types of a given workspace, together with additional
     * information.
     *
     * @return
     * @throws Exception
     */
    @Deprecated
    @GET
    @Produces(MediaType.APPLICATION_JSON)
    @Path("/featuretypes")
    public static Response getFeatureTypes(
            @Context SecurityContext sec,
            @Context ServletContext ctx,

            @PathParam("workspace") String workspace) throws Exception {

        List<GeorepoFeatureType> featureTypes = new ArrayList<GeorepoFeatureType>();
        JDBCConnectionJNDI jdbcMeta = null;
        ResultSet rs = null;

        if (true) {
            // return empty list
            return Response.ok().entity(JsonConverter.toJson(featureTypes)).build();
        }

        try {
            jdbcMeta = new JDBCConnectionJNDI("jdbc/georepoMetaResource");

            GeoserverLayer layer = new GeoserverLayer(ctx);
            GeorepoFeature f = new GeorepoFeature(
                    layer.getGeoserverUrl(),
                    layer.getHeaders(),
                    layer.getGeoserverUser(),
                    layer.getGeoserverPass());

            GeoserverWorkspace wrkspc = new GeoserverWorkspace(
                    layer.getGeoserverRestUrl(),
                    layer.getHeaders(),
                    layer.getGeoserverUser(),
                    layer.getGeoserverPass());

            List<String> datasources = wrkspc.getDatasources(workspace);

            for (String d : datasources) {

                try {

                    System.out.println("\n\nDatasource: " + d);

                    List<FeatureType> ftl1 = f.getFeatureTypes(layer.getGeoserverRestUrl(), workspace, d);

                    for (FeatureType ft : ftl1) {
                        System.out.println("FeatureType: " + ft.getName());

                        String ftName = ft.getName();

                        GeorepoFeatureType featureType = new GeorepoFeatureType();
                        featureType.setName(ftName);
                        featureType.setDatasourceName(d);

                        if (!d.startsWith(GeorepoDatasource.georepoDatasourcePrefix)) {
                            throw new IllegalArgumentException("Datasource " + d + " does not contain correct prefix.");
                            // Integer layerId =
                            // Integer.valueOf(d.split(GeorepoLayer.georepoDatasourcePrefix)[1]);
                        }

                        Integer layerId = -1;
                        String[] ft_parts = ftName.split("_");
                        try {
                            layerId = Integer.valueOf(ft_parts[ft_parts.length - 1]);
                        } catch (NumberFormatException nfe) {
                            continue;
                        }

                        if (layerId < 1) {
                            continue;
                        }

                        String sql = "SELECT layer, lastupdated from datasets WHERE id=?;";
                        System.out.println(sql);

                        PreparedStatement ps = jdbcMeta.prepareStatement(sql);
                        ps.setInt(1, layerId);

                        rs = ps.executeQuery();
                        String lastUpdated = "-";
                        String layerTitle = "-";
                        if (rs.next()) {
                            layerTitle = rs.getString("layer");
                            lastUpdated = rs.getString("lastupdated");
                        }

                        featureType.setLastUpdated(lastUpdated);
                        featureType.setLayerTitle(layerTitle);
                        featureType.setLayerId(layerId);

                        featureTypes.add(featureType);
                    }

                    // }

                } catch (Exception e) {
                    System.out.println("Cannot fetch layer name for datasource " + d + " on workspace " + workspace
                            + ". Continue with next layer...");
                    e.printStackTrace();
                }

            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (jdbcMeta != null) {
                jdbcMeta.closeAll();
                jdbcMeta = null;
            }
        }

        return Response.ok().entity(JsonConverter.toJson(featureTypes)).build();
    }
}
