package de.narimo.georepo.server.api;

import java.util.List;

import javax.servlet.ServletContext;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.SecurityContext;
import javax.ws.rs.ext.Provider;

import de.narimo.commons.json.JsonConverter;
import de.narimo.georepo.server.GeorepoDatasource;
import de.narimo.georepo.server.GeorepoTools;
import de.narimo.georepo.server.dto.FeatureType;
import de.narimo.georepo.server.geoserver.GeoserverLayer;
import de.narimo.georepo.server.geoserver.GeoserverWorkspace;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;

/**
 * georepo.com layer API
 *
 * @author Ulrich Mann
 *
 */
@Provider
@Path("/workspaces/{workspace}/layers")
@Api(value = "LayerController")
public class LayerController {

    public static String geoserverDataDir = "GEOSERVER_DATA_DIR";

    @GET
    @Path("")
    @Produces(MediaType.APPLICATION_JSON)
    @ApiOperation(value = "Retrieve workspace layers")
    public static Response getWorkspaceLayers(
            @Context SecurityContext sec,
            @Context ServletContext ctx,
            @PathParam("workspace") String workspace) {

        try {
            GeoserverLayer layer = new GeoserverLayer(ctx);
            GeoserverWorkspace gworkspace = new GeoserverWorkspace(layer.getGeoserverRestUrl(), layer.getHeaders(),
                    layer.getGeoserverUser(), layer.getGeoserverPass());

            List<FeatureType> layers = gworkspace.getWorkspaceFeatureTypes(workspace);

            return Response.ok().entity(JsonConverter.toJson(layers)).build();
        } catch (Exception e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }

    /**
     * Prerequisites for csv import:
     * - only point data supported yet
     * - column lon for longitude coordinates
     * - column lat for latitude coordinates
     * - filename must only contain alphanumerics, dashes and underscores
     * - filename must end .csv (postgres insert function restriction)
     * - only allowed csvdelimiter is semicolon (;)
     * - currently only utf8 charset allowed, i.e., no umlauts, ß, etc. in csv
     * content
     * - numeric values in csv are inserted as text to database
     *
     * TODO: check: do NOT allow feature type names containing whitespaces! This
     * will mess up queries that are not correctly url encoded or even then.
     *
     * NB: Conventions for new layers, datasources and workspaces:
     * - datasource starts with prefix gd_ and concatenates workspace name
     * - layer name (not layer title!) must not contain whitespaces and end with
     * _gd_ plus the newly created layerId
     *
     * @param sec
     * @param ctxt
     * @param workspace
     * @param sridAsString
     * @param delimiter
     * @param fileName
     * @param readableLayerName
     * @param fileType
     * @param optColumnTypes
     * @return
     * @throws Exception
     */
    @POST
    @Path("")
    public static Response createLayer(
            @Context ServletContext ctx,

            @PathParam("workspace") String workspace,

            @QueryParam("srid") String sridAsString, // EPSG code, defaults to
                                                     // 4326
            @QueryParam("delimiter") String delimiter, // necessary for geocsv,
                                                       // defaults to ;
            @QueryParam("filename") String fileName, // should be the name as
                                                     // how it lives in
                                                     // geoserver_data_dir
            @QueryParam("layername") String readableLayerName,
            @QueryParam("filetype") String fileType,
            @QueryParam("columntypes") String optColumnTypes // optional
                                                             // comma-separated
                                                             // string with
                                                             // postgres column
                                                             // types, e.g.
                                                             // real,text,text,integer...
    ) throws Exception {

        System.out.println("Column types to use: " + optColumnTypes);

        if (fileType.equals("geocsv")) {
            return GeorepoTools.importGeoCSV(
                    ctx, ctx.getInitParameter(geoserverDataDir), sridAsString, delimiter, fileName, readableLayerName,
                    workspace, optColumnTypes);
        } else {
            throw new IllegalArgumentException("Parameter filetype must specify a supported filetype.");
        }

    }

    @POST
    @Path("/{datasource}/update")
    public static Response updateLayerContent(
            @Context ServletContext ctx,

            @PathParam("workspace") String workspace,
            @PathParam("datasource") String layerName,

            @QueryParam("srid") String sridAsString, // EPSG code, defaults to
                                                     // 4326
            @QueryParam("delimiter") String delimiter, // necessary for geocsv,
                                                       // defaults to ;
            @QueryParam("filename") String fileName, // should be the name as
                                                     // how it lives in
                                                     // geoserver_data_dir,
                                                     // csv!!
            // @QueryParam("layername") String readableLayerName,
            @QueryParam("filetype") String fileType,
            @QueryParam("columntypes") String optColumnTypes // optional
                                                             // comma-separated
                                                             // string with
                                                             // postgres column
                                                             // types, e.g.
                                                             // real,text,text,integer...
    ) throws Exception {

        Integer layerId = null;
        try {
            layerId = Integer.valueOf(layerName.substring(layerName.lastIndexOf("_") + 1));
        } catch (Exception e) {
            throw new IllegalArgumentException("Invalid layer name " + layerName + " given. Must be in format "
                    + GeorepoDatasource.georepoDatasourcePrefix + "xxxx.");
        }

        if (fileType.equals("geocsv")) {
            return GeorepoTools.replaceGeoCSVLayer(
                    ctx,
                    ctx.getInitParameter(geoserverDataDir),
                    sridAsString,
                    delimiter,
                    fileName,
                    workspace,
                    optColumnTypes,
                    layerId);
        } else {
            throw new IllegalArgumentException("Parameter filetype is not a supported filetype: " + fileType);
        }

    }

    @DELETE
    @Path("/{layerid}")
    public static Response removeLayer() {

        // TODO

        return Response.status(Status.NOT_IMPLEMENTED).build();
    }

    public static void importFile0(ServletContext ctx, String workspace, String fileName,
            String layerName, String optColumnTypes) throws Exception {

        String epsg = "4326"; // currently expected to be wgs84
        String csvDelimiter = ";";

        String fileType = "";
        String fileExt = fileName.substring(fileName.lastIndexOf(".") + 1);
        System.out.println("importing: " + fileName);
        System.out.println("fileextension: " + fileExt);
        if (fileExt.equals("csv")) {
            fileType = "geocsv";
        }

        LayerController.createLayer(ctx, workspace, epsg, csvDelimiter, fileName, layerName, fileType,
                optColumnTypes);

    }
}
