package de.narimo.georepo.server.api;

import java.io.IOException;

import javax.servlet.ServletContext;
import javax.ws.rs.ForbiddenException;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.SecurityContext;
import javax.ws.rs.ext.Provider;

import de.narimo.commons.dto.geometa.User;
import de.narimo.geocore.ws.repository.UserRepository;
import de.narimo.georepo.server.layer.FeatureLayer;
import de.narimo.georepo.server.repository.DifftableRepository;
import io.swagger.annotations.Api;

@Provider
@Path("/workspaces/{workspace}/layers/{layername}/features")
@Api(value = "FeatureController")
public class FeatureController {

    @GET
    @Produces(MediaType.APPLICATION_JSON)
    @Path("/")
    public static Response getFeatures(
            // should be determined from layerid
            @PathParam("workspace") String dataWorkspace,
            // should be determined from layerid
            @PathParam("layername") String dataLayerName, @Context SecurityContext sec, @Context ServletContext ctx) {

        try {

            // Features may not be retrieved from a diff table using this
            // endpoint
            // That is, double check if the requested workspace is a diff
            // workspace, then disallow!
            boolean isDiffWorkspace = isDiffWorkspace(dataWorkspace, dataLayerName);
            if (isDiffWorkspace) {
                throw new ForbiddenException("The given workspace is a diff workspace.");
            }

            User user = UserRepository.getUser(sec.getUserPrincipal().getName()).get();
            String jsonFeatureCollection = FeatureLayer.getFeatures(user.getId(), dataWorkspace, dataLayerName);

            if (jsonFeatureCollection == null) {
                throw new InternalError("Could not get feature changes.");
            }

            return Response.ok().entity(jsonFeatureCollection).build();
        } catch (IOException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        } catch (ForbiddenException e) {
            e.printStackTrace();
            return Response.status(Status.FORBIDDEN).build();
        } catch (InternalError | RuntimeException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }

    /**
     * Is a workspace actually a diff workspace?
     * 
     * @param dataWorkspace
     * @param dataLayerName
     * @return
     */
    private static boolean isDiffWorkspace(String dataWorkspace, String dataLayerName) {
        String diffWorkspace = DifftableRepository.getDiffWorkspace(dataWorkspace);
        String diffLayerName = DifftableRepository.getDiffLayername(dataWorkspace, diffWorkspace, dataLayerName);
        return dataWorkspace.equals(diffWorkspace) || dataLayerName.equals(diffLayerName);
    }
}
