package de.narimo.georepo.server.api;

import java.io.IOException;
import java.io.InputStream;
import java.sql.SQLException;

import javax.naming.OperationNotSupportedException;
import javax.servlet.ServletContext;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.SecurityContext;
import javax.ws.rs.ext.Provider;

import com.fasterxml.jackson.databind.ObjectMapper;

import de.narimo.commons.dto.geometa.User;
import de.narimo.geocore.ws.repository.UserRepository;
import de.narimo.georepo.server.geoserver.GeoserverLayer;
import de.narimo.georepo.server.geoserver.GeoserverWorkspace;
import de.narimo.georepo.server.providers.WorkspaceRegistrationDetails;
import de.narimo.georepo.server.providers.WorkspaceRegistrationProvider;
import de.narimo.georepo.server.repository.DifftableRepository;
import de.narimo.georepo.server.repository.WorkspaceRepository;
import de.narimo.georepo.server.tools.AdminTools;
import de.narimo.georepo.server.tools.WorkspacePermissions;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;

@Provider
@Path("/workspaces/{workspace}")
@Api(value = "WorkspaceController")
public class WorkspaceController {

    /**
     * Returns 200 OK when the workspace authorization filter confirmed, that we
     * have read (=GET) permission for the requested workspace.
     * 
     * @param workspace
     * @param sec
     * @param body
     * @return
     */
    @GET
    @Path("")
    @ApiOperation(value = "Checks workspace read permission")
    public static Response checkWorkspacePermission(@Context SecurityContext sec,
            @PathParam("workspace") String workspace,
            InputStream body) {

        return Response.ok().build();
    }

    @POST
    @Path("/register")
    @ApiOperation(value = "Register for an existing workspace")
    public static Response registerUserWorkspace(
            @PathParam("workspace") String workspace,
            @Context SecurityContext sec,
            InputStream body) {

        try {
            User user = UserRepository.getUser(sec.getUserPrincipal().getName()).get();
            ObjectMapper mapper = new ObjectMapper();

            WorkspaceRegistrationDetails registrationDetails = mapper.readValue(body,
                    WorkspaceRegistrationDetails.class);
            registrationDetails.setWorkspace(workspace);

            WorkspaceRegistrationProvider.addWorkspaceUser(user, registrationDetails);

            return Response.ok().build();
        } catch (IOException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        } catch (InternalError | RuntimeException | SQLException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }

    @DELETE
    @Path("")
    @ApiOperation(value = "Delete an existing workspace")
    public static Response deleteWorkspace(
            @Context SecurityContext sec,
            @Context ServletContext ctx,
            @PathParam("workspace") String workspace) throws OperationNotSupportedException {

        if (true) {
            return Response.status(Status.NOT_IMPLEMENTED).build();
        }

        return Response.ok().build();
    }

    @GET
    @Path("/services/{service}/settings")
    public static Response getWorkspaceSettings(
            @Context SecurityContext sec,
            @Context ServletContext ctx,

            @PathParam("service") String service,
            @PathParam("workspace") String workspace) throws Exception {

        GeoserverLayer layer = new GeoserverLayer(ctx);
        GeoserverWorkspace ws = new GeoserverWorkspace(
                workspace,
                layer.getGeoserverRestUrl(),
                layer.getHeaders(),
                layer.getGeoserverUser(),
                layer.getGeoserverPass());

        String wsSettings = ws.getWorkspaceServiceSettings(service);

        return Response.ok().entity(wsSettings).build();
    }

    @PUT
    @Path("/services/{service}/settings")
    public static Response setWorkspaceSettings(
            @Context SecurityContext sec,
            @Context ServletContext ctx,

            @PathParam("service") String service,
            @PathParam("workspace") String workspace,

            @QueryParam("enabled") String active,
            @QueryParam("title") String title,
            @QueryParam("description") String description,
            @QueryParam("maintainer") String maintainer,
            @QueryParam("fees") String fees,
            @QueryParam("accessConstraints") String accessConstraints,
            @QueryParam("onlineResource") String onlineResource) throws Exception {

        GeoserverLayer layer = new GeoserverLayer(ctx);
        GeoserverWorkspace ws = new GeoserverWorkspace(
                workspace,
                layer.getGeoserverRestUrl(),
                layer.getHeaders(),
                layer.getGeoserverUser(),
                layer.getGeoserverPass());

        boolean enabled = false;
        try {
            enabled = Boolean.parseBoolean(active);
        } catch (Exception e) {
            e.printStackTrace();
        }

        ws.modifyWorkspaceOWSSettings(service, title, description, maintainer, fees,
                accessConstraints, onlineResource, null, enabled);

        return Response.ok("Workspace " + workspace + " has been modified with the given settings.").build();
    }

    /**
     * Overwrite all current permission settings for the given workspace and user
     * with the new permission.
     *
     * A user with admin permissions on the workspace is allowed to change
     * permissions on the same workspace for other users.
     *
     * @param ctx
     * @param ws
     * @param username
     * @param perm
     * @return
     * @throws SQLException
     */
    @PUT
    @Path("/permissions/{permission}")
    public static Response setWorkspacePermissionEndpoint(
            @Context ServletContext ctx,
            @Context SecurityContext sec,
            @PathParam("workspace") String ws,
            @PathParam("permission") String perm,
            InputStream body // User object should contain user mail/ username
    ) {

        try {
            User admin = UserRepository.getUser(sec.getUserPrincipal().getName()).get();
            User user = new ObjectMapper().readValue(body, User.class);
            setWorkspacePermission(admin, user, ws, perm);
        } catch (IOException e) {
            return Response.status(Status.BAD_REQUEST)
                    .entity("Permissions for workspace " + ws + " could not be modified.").build();
        } catch (SQLException e) {
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
        return Response.ok("Permissions for workspace " + ws + " have been modified with the given settings.").build();
    }

    @GET
    @Path("/managed")
    @ApiOperation(value = "Retrieve a diff workspace")
    public static Response getDiffWorkspace(@Context SecurityContext sec,
            @PathParam("workspace") String dataWorkspace,
            InputStream body) {

        String diffWorkspace = DifftableRepository.getDiffWorkspace(dataWorkspace);
//        List<String> diffWorkspaces = new ArrayList<>();
//        diffWorkspaces.add(diffWorkspace);

        String diffWorkspaceJson = "{\"diffWorkspace\":\"" + diffWorkspace + "\"}";
        return Response.ok().entity(diffWorkspaceJson).build();
    }

    /**
     * Set a workspace permission.
     *
     * @param admin
     * @param user
     * @param ws
     * @param permission
     * @throws IOException
     * @throws SQLException
     */
    public static void setWorkspacePermission(User admin, User user, String ws, String permission)
            throws IOException, SQLException {
        AdminTools.checkAdminPermission(admin.getId(), ws);
        setWorkspacePermissions0(ws, user.getName(), permission);
    }

    /**
     * NOTE: Usage of this method is discouraged.
     * 
     * It should only be used to set the initial permission for the creator of the
     * workspace.
     * 
     * @param admin
     * @param user
     * @param ws
     * @throws IOException
     * @throws SQLException
     */
    static void setInitialWorkspacePermission(User user, String ws)
            throws IOException, SQLException {
        setWorkspacePermissions0(ws, user.getName(), WorkspacePermissions.adminPermission);
    }

    /**
     *
     * All entries from workspaces are cleared for the given combination of userid,
     * workspace and permissions. Then the new entry is written. So there is always
     * only a single unique setting for those combinations.
     *
     * @param workspace
     * @param username
     * @param permission
     *
     * @throws IOException
     * @throws SQLException
     */
    private static void setWorkspacePermissions0(String workspace, String username, String permission)
            throws IOException, SQLException {

        if (workspace == null) {
            throw new IllegalArgumentException("Workspace must not be null;");
        }

        if (!WorkspacePermissions.availablePermissions.contains(permission)) {
            throw new IllegalArgumentException("Unknown workspace permission " + permission);
        }

        Integer userid = UserRepository.getUserId(username);
        if (userid == null) {
            throw new IllegalArgumentException("Cannot set workspace permission for userid " + userid + ";");
        }

        WorkspaceRepository.changeWorkspacePermission(userid, workspace, permission);
    }
}
