package de.narimo.georepo.server.api;

import java.io.IOException;
import java.sql.SQLException;
import java.util.Map;

import javax.servlet.ServletContext;
import javax.ws.rs.ForbiddenException;
import javax.ws.rs.GET;
import javax.ws.rs.OPTIONS;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.SecurityContext;
import javax.ws.rs.ext.Provider;

import de.narimo.commons.dto.geometa.User;
import de.narimo.geocore.ws.repository.UserRepository;
import de.narimo.georepo.server.db.GeorepoConstants;
import de.narimo.georepo.server.geoserver.Contact;
import de.narimo.georepo.server.geoserver.GeoserverLayer;
import de.narimo.georepo.server.geoserver.GeoserverWorkspace;
import de.narimo.georepo.server.repository.WorkspaceRepository;
import de.narimo.georepo.server.tools.AdminTools;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;

@Provider
@Path("/workspaces")
@Api(value = "WorkspacesController")
public class WorkspacesController {

    /**
     * Allows a browser to issue a CORS preflight request.
     * 
     * Generic response for OPTIONS request currently does not support setting and
     * CORS headers.
     *
     * Application-specific OPTIONS responses may be set by the application itself
     * or at sub-paths.
     *
     * @param sec
     * @param ctx
     * @return
     * @throws Exception
     */
    @OPTIONS
    @Path("")
    public static Response optionsLayerRequest(@Context SecurityContext sec, @Context ServletContext ctx) {
        return Response.ok().build();
    }

    @GET
    @Path("")
    @Produces(MediaType.APPLICATION_JSON)
    @ApiOperation(value = "Retrieve workspaces")
    public static Response getWorkspaces(@Context SecurityContext sec, @Context ServletContext ctx) {

        try {
            String username = sec.getUserPrincipal().getName();
            int userId = UserRepository.getUserId(username);

            Map<String, String> workspaces = WorkspaceRepository.getWorkspaces(userId, false);
            return Response.ok().entity(workspaces).build();
        } catch (Exception e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }

    @POST
    @Path("/")
    @ApiOperation(value = "Creates a new workspace")
    public static Response createWorkspace(@Context SecurityContext sec, @Context ServletContext ctx) {

        String username = sec.getUserPrincipal().getName();
        User user = UserRepository.getUser(username).get();

        String workspaceName = null;
        try {
            workspaceName = createNewWorkspace(ctx, user);
            WorkspaceController.setInitialWorkspacePermission(user, workspaceName);
        } catch (ForbiddenException e) {
            System.out.println("User " + username + " has no permission to create an additional workspace.");
            return Response.status(403).entity("User has no permission to create additional workspace.").build();
        } catch (IOException | SQLException e) {
            System.out.println("An issue occured while creating workspace.");
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }

        return Response.status(201).entity("Workspace " + workspaceName + " has been created.").build();
    }

    public static String createNewWorkspace(ServletContext ctx, User user) throws IOException, SQLException {
        return createNewWorkspace0(ctx, user, false);
    }

    public static String createDiffWorkspace(ServletContext ctx, User user) throws IOException, SQLException {
        return createNewWorkspace0(ctx, user, true);
    }

    /**
     * Creates a workspace with a naming that is not yet taken. Does not yet set
     * permissions on that new workspace.
     * 
     * @param ctx
     * @param user
     * @return
     * @throws IOException
     * @throws SQLException
     */
    public static String createNewWorkspace0(ServletContext ctx, User user, boolean isDiffWorkspace)
            throws IOException, SQLException {

        if (user.getId() == null) {
            throw new IllegalArgumentException("Could not create a workspace for user " + user.getName()
                    + ". Unknown user " + user.getName() + ".");
        }

        if (!isDiffWorkspace) {
            AdminTools.checkCanCreateWorkspace(user.getId());
        } else {
            // do not check against user's workspace count for diff worksapces, since the
            // diff workspace is created automatically
        }

        String newWorkspace = getNewWorkspaceName();
        GeoserverLayer layer = new GeoserverLayer(ctx);
        while (layer.workspaceExists(layer.getWorkspacesXML(), newWorkspace)) {
            // re-create a workspace name which is not yet taken
            newWorkspace = getNewWorkspaceName();
        }

        GeoserverWorkspace ws = new GeoserverWorkspace(newWorkspace, layer.getGeoserverRestUrl(), layer.getHeaders(),
                layer.getGeoserverUser(), layer.getGeoserverPass());
        ws.createWorkspace();

        // add default workspace settings
        Contact georepoContact = new Contact();
        georepoContact.setAddress("Bertolt-Brecht-Allee 24");
        georepoContact.setAddressCity("Dresden");
        georepoContact.setAddressCountry("Germany");
        georepoContact.setAddressPostalCode("01309");
        georepoContact.setAddressType("Postal");
        georepoContact.setAddressElectronicMailAddress("info@georepo.com");
        georepoContact.setContactEmail("info@georepo.com");
        georepoContact.setContactOrganization("narimo systems");
        georepoContact.setContactPosition("Service provider");
        georepoContact.setContactVoice("+49 351 21359935");

        String proxyBaseUrl = ctx.getInitParameter("GEOREPO_URL_EXT") + ctx.getInitParameter("GEOREPO_API_BASE_PATH")
                + "/workspaces";
        ws.modifyWorkspaceSettings("UTF-8", proxyBaseUrl, georepoContact, "https://georepo.com", true);

        // Create WMS and WFS services for that workspace, which must be enabled
        // separately
        ws.createWFSService();
        ws.createWMSService();

        return newWorkspace;
    }

    public static String getNewWorkspaceName() {
        long workspaceCount = Math.round(Math.random() * 10000);
        return GeorepoConstants.WORKSPACEPREFIX + workspaceCount;
    }
}
