package de.narimo.georepo.server.repository;

import java.io.IOException;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.ws.rs.NotFoundException;

import org.geotools.feature.FeatureCollection;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryFactory;
import org.opengis.feature.simple.SimpleFeature;

import de.narimo.commons.jdbc.JDBCConnectionJNDI;
import de.narimo.georepo.server.db.GeorepoConstants;
import de.narimo.georepo.server.layer.FeatureBuilder;
import de.narimo.georepo.server.tools.GeometryTools;
import de.narimo.georepo.server.tools.QueryCheck;

public class DatasetRepository extends ParentDatasetRepository {

    public static FeatureCollection getDatasets(String dataTableName, long userId) throws IOException {

        QueryCheck.checkTableName(dataTableName);

        JDBCConnectionJNDI jdbcData = null;

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            Map<String, String> columnsAndTypes = getColumnTypes(dataTableName,
                    GeorepoConstants.DEFAULTSCHEMA);

            Map<String, Class> propertyTypes = new HashMap<String, Class>();

            for (String colName : columnsAndTypes.keySet()) {
                String colType = columnsAndTypes.get(colName);
                if (colType.equals("integer")) {
                    propertyTypes.put(colName, Integer.class);
                } else if (colType.equals("double precision")) {
                    propertyTypes.put(colName, Double.class);
                } else {
                    propertyTypes.put(colName, String.class);
                }
            }

            String query = getDatasetsQuery(dataTableName);
            ResultSet rs = jdbcData.executeQuery(query);

            FeatureBuilder fb = new FeatureBuilder(propertyTypes);
            GeometryFactory gf = new GeometryFactory();

            List<SimpleFeature> simpleFeatures = new ArrayList<>();

            while (rs.next()) {
                double x = rs.getDouble("st_x");
                double y = rs.getDouble("st_y");

                Geometry geometry = null;
                if (!rs.wasNull()) {
                    geometry = GeometryTools.createPointGeometry(x, y, gf);
                }
                Map<String, Object> properties = new HashMap<>();
                for (String colName : columnsAndTypes.keySet()) {
                    properties.put(colName, rs.getString(colName));
                }

                SimpleFeature simpleFeature = fb.createSimpleFeature(geometry, properties);
                simpleFeatures.add(simpleFeature);
            }
            return fb.createFeatureCollection(simpleFeatures);

        } catch (SQLException e) {
            e.printStackTrace();
        } catch (Exception e) {
            throw new IOException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
        return null;
    }

    private static String getDatasetsQuery(String tableName) {
        return "SELECT ST_X(the_geom), ST_Y(the_geom), * from public." + tableName + ";";
    }

    public static int getLayerId(String workspace, String featureType) {

        JDBCConnectionJNDI jdbcMeta = null;

        try {
            jdbcMeta = new JDBCConnectionJNDI("jdbc/georepoMetaResource");

            String sql = "SELECT id from public.datasets WHERE workspace = ? AND featuretype = ?;";
            System.out.println(sql);

            PreparedStatement ps = jdbcMeta.prepareStatement(sql);
            ps.setString(1, workspace);
            ps.setString(2, featureType);

            ResultSet rs = ps.executeQuery();

            int count = 0;
            Integer id = null;
            while (rs.next()) {
                id = rs.getInt("id");
                count++;
            }

            if (count > 1) {
                throw new NotFoundException(
                        "Could not determine distinct layer id. Please review datasets table consistency for workspace "
                                + workspace + " and feature type " + featureType + "!");
            }

            return id;
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (jdbcMeta != null) {
                jdbcMeta.closeAll();
                jdbcMeta = null;
            }
        }
        throw new NotFoundException("Could not determine layer id.");
    }
}
