package de.narimo.georepo.server.tools;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStreamWriter;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.Arrays;
import java.util.List;

import javax.servlet.ServletContext;

import com.csvreader.CsvReader;

import de.narimo.commons.http.URLResponse;
import de.narimo.commons.ws.http.HttpMethod;
import de.narimo.commons.ws.http.HttpURLClient;
import de.narimo.georepo.server.api.LayerController;

public class ImportTools {

    /**
     * Import file that exists on the server.
     * 
     * @param ctx
     * @param workspace
     * @param fileName
     * @param layerTitle
     * @param optColumnTypes
     * @throws Exception
     */
    public static void importCsvFile(ServletContext ctx, String workspace, String fileName, String layerTitle,
            String optColumnTypes) throws Exception {

        String srid = "4326"; // currently expected to be wgs84
        String csvDelimiter = ";";

        String fileType = "";
        String fileExt = fileName.substring(fileName.lastIndexOf(".") + 1);
        System.out.println("importing: " + fileName);
        System.out.println("fileextension: " + fileExt);
        if (fileExt.equals("csv")) {
            fileType = "geocsv";
        }

        LayerController.createLayer(ctx, workspace, srid, csvDelimiter, fileName, layerTitle, fileType, optColumnTypes);
    }

    /**
     * Import file from a remote http(s) resource.
     * 
     * @param ctx
     * @param layerTitle
     * @param optColumnTypes
     * @param workspace
     * @param remoteCSVUrl
     * @param usedCSVDelimiter
     * @param decimalSep
     * @param thousandsSep
     * @param optLongitudeName
     * @param optLatitudeName
     * @param approved
     * @param keepRowColumnName
     * @param keepRowColumnValue
     * @throws Exception
     */
    public static void importRemoteFile(ServletContext ctx, String layerTitle, String optColumnTypes, String workspace,
            String remoteCSVUrl, String usedCSVDelimiter, String decimalSep, String thousandsSep, String optLongitudeName,
            String optLatitudeName, boolean approved, String keepRowColumnName, String keepRowColumnValue) throws Exception {

        String srid = "4326"; // currently expected to be wgs84
        String wantedCSVDelimiter = ";";

        List<String> columnTypes = optColumnTypes == null ? null : Arrays.asList(optColumnTypes.split("\\s*,\\s*"));
        File csvFile = ImportTools.getFreshCSVFile(ctx, workspace, remoteCSVUrl, usedCSVDelimiter, wantedCSVDelimiter,
                columnTypes, decimalSep, thousandsSep, optLongitudeName, optLatitudeName, approved, keepRowColumnName,
                keepRowColumnValue);

        System.out.println("Importing csv file... ");

        LayerController.createLayer(ctx, workspace, srid, wantedCSVDelimiter, csvFile.getName(), layerTitle, "geocsv",
                optColumnTypes);

        Files.delete(csvFile.toPath());
        System.out.println("Removed csv input file successfully... ");
    }

    /**
     * Retrieve a file from a remote server
     * 
     * @param ctx
     * @param workspace
     * @param remoteCSVUrl
     * @param fromDelimiter
     * @param toDelimiter
     * @param columnTypes
     * @param decimalSep
     * @param thousandsSep
     * @param optLongitudeName
     * @param optLatitudeName
     * @param approved
     * @param keepRowColumnName
     * @param keepRowColumnValue
     * @return
     * @throws Exception
     */
    public static File getFreshCSVFile(ServletContext ctx, String workspace, String remoteCSVUrl, String fromDelimiter,
            String toDelimiter, List<String> columnTypes, String decimalSep, String thousandsSep, String optLongitudeName,
            String optLatitudeName, boolean approved, String keepRowColumnName, String keepRowColumnValue) throws Exception {

        InputStream in = null;
        File csvFile = null;

        try {
            if (!remoteCSVUrl.endsWith(".csv") && !approved) {
                throw new IllegalArgumentException("Remote URL to CSV file must end with \".csv\".");
            }
            System.out.println("Getting data from " + remoteCSVUrl + ".");

            URLResponse rr = HttpURLClient.sendRequest(remoteCSVUrl, null, null, null, HttpMethod.GET, null, null, null,
                    remoteCSVUrl.startsWith("https"), true);

            in = rr.getResponseStream();

            if (in == null) {
                throw new IOException("CSV input file is null");
            }
            System.out.println("Saving CSV data to file... ");

            String fileName = "remote_csv_" + System.currentTimeMillis() + ".csv";
            csvFile = new File(ctx.getInitParameter("GEOSERVER_DATA_DIR") + "/data/" + workspace + "/upload/" + fileName);
            csvFile.mkdirs(); // should be created by geoserver API actually on setting up new workspace
            Files.copy(in, csvFile.toPath(), StandardCopyOption.REPLACE_EXISTING);

            // TODO: do not proceed for empty files?
            // maybe do process loggin, which files/ parts failed with info about empty file

            ImportTools.cleanCSVFile(csvFile, fromDelimiter, toDelimiter, columnTypes, decimalSep, thousandsSep, optLongitudeName,
                    optLatitudeName, keepRowColumnName, keepRowColumnValue);

        } finally {
            if (in != null) {
                in.close();
            }
        }
        return csvFile;
    }

    /**
     * 
     * Do some cleaning operations on a csv file, using CSVReader. 
     * 1) remove @ in column names 
     * 2) change artificial delimiter -;;- to ; and escape correctly
     * 
     * NOTE: Be aware on changes, this must fit to all CSV file, especially OSM Overpass files. Header names must be unique!
     * 
     * @param csvFile
     * @param fromDelimiter
     * @param toDelimiter
     * @param columnTypes
     * @param decimalSep
     * @param thousandsSep
     * @param optLongitudeName
     * @param optLatitudeName
     * @throws IOException
     */
    public static void cleanCSVFile(
            File csvFile, 
            String fromDelimiter, 
            String toDelimiter, 
            List<String> columnTypes, 
            String decimalSep, 
            String thousandsSep,
            String optLongitudeName,
            String optLatitudeName,
            String keepRowColumnName,
            String keepRowColumnValue
            ) throws IOException{

        if(decimalSep!=null && decimalSep.trim().equals("")) decimalSep=null;
        if(decimalSep!=null && (!decimalSep.equals(".") || !decimalSep.equals(","))) throw new IOException("Supported decimal separator for numbers is '.' or ','.");
        
        if(thousandsSep!=null && thousandsSep.trim().equals("")) thousandsSep=null;
        if(thousandsSep!=null && (!thousandsSep.equals(".") || !thousandsSep.equals(","))) throw new IOException("Supported thousands separator for numbers is '.' or ','.");
        
        CsvReader r=null;
        BufferedWriter bw = null;
                
        String csvFileName = csvFile.getName();
        String name = csvFileName.substring(0, csvFileName.lastIndexOf("."));
        String extension = csvFileName.substring(csvFileName.lastIndexOf("."), csvFileName.length());
        String toFile = name+"_tmp"+extension;
        File outFile = new File(csvFile.getParent()+"/"+toFile);
        
        System.out.println("creating tmp file: "+outFile);
        
        try  {
            r = new CsvReader(csvFile.getAbsolutePath(), fromDelimiter.charAt(0));
            r.readHeaders();
            r.setSkipEmptyRecords(true);
            r.setTrimWhitespace(true);
            
            FileOutputStream fs = new FileOutputStream(outFile);
            OutputStreamWriter sr = new OutputStreamWriter(fs, "utf-8");
            bw = new BufferedWriter(sr);

            System.out.println("Column types: "+columnTypes.toString());            
            
            String[] headers = r.getHeaders();
            String[] newHeaders = new String[headers.length];
            
            System.out.println("Replacing headers:\n"+Arrays.toString(headers));
            Integer keepRowColumnIndex = null;
            int i=0;
            for(String header : headers){
                if(header.equals(keepRowColumnName)){
                    keepRowColumnIndex = i;
                }
                header = header.replace("@", "");
                header = header.replace(":", "_");              
                if(optLatitudeName!=null) header = header.replace(optLatitudeName, "lat"); //lat is the expected latitude column identifier
                if(optLongitudeName!=null) header = header.replace(optLongitudeName, "lon"); //lat is the expected latitude column identifier
                
                newHeaders[i] = header;
                i++;
            }
            System.out.println("With:\n"+Arrays.toString(newHeaders));
            
            StringBuilder sb = new StringBuilder();         
            int x=0;
            for (String s : newHeaders) {
                if(x==0) sb.append(s);
                else sb.append(toDelimiter).append(s);
                x++;
            }
            bw.write(sb.toString()+"\n");

            while(r.readRecord()){
                String line = cleanRow(r.getValues(), toDelimiter, columnTypes, thousandsSep, keepRowColumnIndex, keepRowColumnValue);
                if(line!=null){
                    bw.write(line+"\n");
                }
            }

        }finally{
            System.out.println("goin to finally");
            bw.close();
        }
        
        //replace csv file
        Files.copy(outFile.toPath(), csvFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
        Files.delete(outFile.toPath());
        
        System.out.println("\n\nCleaning CSV file DONE.\n\n");
    }
    
    /**
     * Cleans a csv row.
     * Returns null if record value at index keepRowColumnIndex does not equal keepRowColumnValue.
     * 
     * Omitting completely empty lines is done by csv reader automatically.
     * Trimming record whitespaces is done by csv reader.
     * 
     * @param values
     * @param toDelimiter
     * @param columnTypes
     * @param thousandsSep
     * @return
     */
    private static String cleanRow(
            String[] values, 
            String toDelimiter, 
            List<String> columnTypes, 
            String thousandsSep,
            Integer keepRowColumnIndex,
            String keepRowColumnValue
            ){
        String line = "";
        for(int j=0; j<values.length; j++){
            String s = values[j];
            
            if(keepRowColumnIndex!=null && j==keepRowColumnIndex){
                if(!s.equals(keepRowColumnValue)) return null;
            }
            
            s = s.contains(toDelimiter) ? "\""+s+"\"" : s; //escape delimiter in csv file

            if(columnTypes!=null){
                if(columnTypes.get(j).equals("integer")){
                    if(thousandsSep!=null) s = s.replace(String.valueOf(thousandsSep), ""); //we don't need thousands separators in integer values
                }
                //TODO: if type is double precision and we have both, decimal and thousands seps, they must not be the same char
                if(columnTypes.get(j).equals("double precision")){}
            }
            line += j==0 ? s : toDelimiter+s;
        }
        return line;
    }
}
