package de.narimo.georepo.server.api;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.sql.SQLException;
import java.util.List;

import javax.imageio.IIOException;
import javax.servlet.ServletContext;
import javax.ws.rs.GET;
import javax.ws.rs.NotFoundException;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.SecurityContext;
import javax.ws.rs.ext.Provider;

import de.narimo.commons.dto.geometa.User;
import de.narimo.geocore.ws.repository.UserRepository;
import de.narimo.georepo.server.api.documents.DocumentType;
import de.narimo.georepo.server.api.documents.FeatureDocument;
import de.narimo.georepo.server.api.documents.ImageAndDocumentService;
import de.narimo.georepo.server.repository.DatasetRepository;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;

@Provider
@Path("/workspaces/{workspace}/layers/{layername}/features/{gfid}/documents")
@Tag(name = "Documents")
public class DocumentController {

    ImageAndDocumentService imageAndDocumentService = new ImageAndDocumentService();

    @GET
    @Path("/")
    @Produces(MediaType.APPLICATION_JSON)
    @Operation(summary = "Retrieves all documents of a feature")
    public Response getFeatureDocuments(
            @Context SecurityContext sec,
            @Context ServletContext ctx,
            @PathParam("workspace") String workspace,
            @PathParam("layername") String layerName,
            @PathParam("gfid") int gfid) {

        try {
            User user = UserRepository.getUser(sec.getUserPrincipal().getName()).get();

            int layerId = DatasetRepository.getLayerId(workspace, layerName);
            List<FeatureDocument> imagesOrDocuments = imageAndDocumentService.getFeatureImagesOrDocuments(user,
                    workspace, layerId, gfid, DocumentType.DOCUMENT);

            return Response.ok().entity(imagesOrDocuments).build();
        } catch (SQLException | InternalError e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        } catch (IOException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).entity("Could not retrieve image names. ").build();
        }
    }

    @GET
    @Path("/{filename}")
    @Produces({ "application/pdf" })
    @Operation(summary = "Retrieve a document of a feature by its name")
    public Response getFeatureDocument(
            @Context SecurityContext sec,
            @Context ServletContext ctx,
            @PathParam("workspace") String workspace,
            @PathParam("layername") String layerName,
            @PathParam("gfid") int gfid,
            @PathParam("filename") String fileName) {

        try {
            int layerId = DatasetRepository.getLayerId(workspace, layerName);
            byte[] fileData = imageAndDocumentService.getFeatureImageOrDocument(ctx, workspace, layerId, gfid,
                    fileName, DocumentType.DOCUMENT);

            return Response.ok(new ByteArrayInputStream(fileData)).build();

        } catch (IIOException | NotFoundException e) {
            return Response.status(Status.NOT_FOUND).build();
        } catch (SQLException | InternalError e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        } catch (IOException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).entity("Could not retrieve image. ").build();
        }
    }

}
