package de.narimo.georepo.server.api;

import java.io.IOException;
import java.io.InputStream;
import java.sql.SQLException;

import javax.servlet.ServletContext;
import javax.ws.rs.Consumes;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.SecurityContext;
import javax.ws.rs.ext.Provider;

import de.narimo.commons.dto.geometa.User;
import de.narimo.geocore.ws.repository.UserRepository;
import de.narimo.georepo.server.api.documents.DocumentType;
import de.narimo.georepo.server.api.documents.ImageAndDocumentService;
import de.narimo.georepo.server.repository.DatasetRepository;
import de.narimo.georepo.server.repository.DifftableRepository;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;

@Provider
@Path("/workspaces/{diffWorkspace}/layers/{layername}/features/{gfid}/documents/changes")
@Tag(name = "Documents")
public class DocumentModificationController {

    ImageAndDocumentService imageAndDocumentService = new ImageAndDocumentService();

    @POST
    @Path("/")
    @Consumes({ "application/pdf" })
    @Operation(summary = "Suggest adding a new document to a feature of a layer")
    public Response postLayerDocument(
            @Context SecurityContext sec,
            @Context ServletContext ctx,
            @PathParam("diffWorkspace") String diffWorkspace,
            @PathParam("layername") String dataLayerName,
            @PathParam("gfid") int gfid,

            @QueryParam("filename") String filename,
            InputStream uploadedInputStream

    ) throws IOException, SQLException {

        try {

            User user = UserRepository.getUser(sec.getUserPrincipal().getName()).get();

            String workspace = DifftableRepository.getDataWorkspace(diffWorkspace, dataLayerName);
            if (workspace == null) {
                throw new InternalError("No data workspace defined.");
            }

            // If user has no write permission on that workspace, he would already have been
            // thrown out by authorization layer
            int layerId = DatasetRepository.getLayerId(workspace, dataLayerName);
            int diffImageId = imageAndDocumentService.addImageOrDocument(ctx, user, workspace, uploadedInputStream,
                    filename,
                    layerId,
                    gfid, DocumentType.DOCUMENT);

            // TODO: #255: this should be moved to own endpoint
            imageAndDocumentService.acceptImageOrDocument(ctx, layerId, gfid, diffImageId, workspace,
                    DocumentType.DOCUMENT);

        } catch (SQLException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        } catch (IOException e) {
            e.printStackTrace();
            if (e.getMessage().contains("File size should not exceed")
                    || e.getMessage().contains("Image dimensions should not exceed")) {
                return Response.status(Status.REQUEST_ENTITY_TOO_LARGE).entity("File size exceeds maximum limit. ")
                        .build();
            }
            return Response.status(Status.BAD_REQUEST).entity("Could not upload document. ").build();
        }

        return Response.status(201).build();
    }
}
