package de.narimo.georepo.server.api;

import java.io.IOException;

import javax.ws.rs.DELETE;
import javax.ws.rs.ForbiddenException;
import javax.ws.rs.NotFoundException;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.SecurityContext;
import javax.ws.rs.ext.Provider;

import de.narimo.geocore.ws.repository.UserRepository;
import de.narimo.georepo.server.layer.DiffFeatureLayer;
import de.narimo.georepo.server.repository.DifftableRepository;
import de.narimo.georepo.server.tools.AdminTools;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;

@Provider
@Path("/workspaces/{workspace}/layers/{layername}/features/changes/{grpfid}")
@Tag(name = "Feature Reviews", description = "Accept or decline modifications that have been suggested by users.")
public class FeatureReviewController {

    @PUT
    @Path("/accept")
    @Operation(summary = "Accept a suggested modification")
    public static Response acceptFeatureChangeset(
            @PathParam("workspace") String diffWorkspace,
            @PathParam("layername") String dataLayerName,
            @PathParam("grpfid") String grpfidIn,
            @Context SecurityContext sec) {

        try {

            Integer grpfidI = Integer.valueOf(grpfidIn);
            if (grpfidI == null) {
                throw new IOException("Invalid feature id: null.");
            }
            int grpfid = grpfidI.intValue();

            String username = sec.getUserPrincipal().getName();
            int userId = UserRepository.getUserId(username);

            String dataWorkspace = DifftableRepository.getDataWorkspace(diffWorkspace, dataLayerName);
            if (dataWorkspace == null) {
                throw new InternalError("No data workspace defined.");
            }

            AdminTools.checkAdminPermission(userId, dataWorkspace);

            DiffFeatureLayer.acceptFeatureChange(grpfid, dataWorkspace, dataLayerName);
            return Response.ok().build();

        } catch (NumberFormatException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).entity("Invalid feature id.").build();
        } catch (NotFoundException e) {
            e.printStackTrace();
            return Response.status(Status.NOT_FOUND).build();
        } catch (IOException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        } catch (ForbiddenException e) {
            e.printStackTrace();
            return Response.status(Status.FORBIDDEN).build();
        } catch (InternalError | RuntimeException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }

    @DELETE
    @Path("/decline")
    @Operation(summary = "Decline a suggested modification")
    public static Response declineFeatureChangeset(
            @PathParam("workspace") String diffWorkspace,
            @PathParam("layername") String dataLayerName,
            @PathParam("grpfid") String grpfidIn,
            @Context SecurityContext sec) {

        try {

            Integer grpfidI = Integer.valueOf(grpfidIn);
            if (grpfidI == null) {
                throw new IOException("Invalid feature id: null.");
            }
            int grpfid = grpfidI.intValue();

            String username = sec.getUserPrincipal().getName();
            int userId = UserRepository.getUserId(username);

            String dataWorkspace = DifftableRepository.getDataWorkspace(diffWorkspace, dataLayerName);
            if (dataWorkspace == null) {
                throw new InternalError("No data workspace defined.");
            }

            AdminTools.checkAdminPermission(userId, dataWorkspace);

            DiffFeatureLayer.declineFeatureChange(grpfid, dataWorkspace, dataLayerName);
            return Response.ok().build();

        } catch (NumberFormatException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).entity("Invalid feature id.").build();
        } catch (NotFoundException e) {
            e.printStackTrace();
            return Response.status(Status.NOT_FOUND).build();
        } catch (IOException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        } catch (ForbiddenException e) {
            e.printStackTrace();
            return Response.status(Status.FORBIDDEN).build();
        } catch (InternalError | RuntimeException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }
}
