package de.narimo.georepo.server.api;

import java.io.IOException;
import java.sql.SQLException;
import java.util.List;

import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.SecurityContext;
import javax.ws.rs.ext.Provider;

import de.narimo.commons.dto.geometa.User;
import de.narimo.geocore.ws.repository.UserRepository;
import de.narimo.georepo.server.api.comments.CommentOrLinkService;
import de.narimo.georepo.server.api.comments.Weblink;
import de.narimo.georepo.server.repository.DatasetRepository;
import de.narimo.georepo.server.tools.AdminTools;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;

@Provider
@Path("/workspaces/{workspace}/layers/{layername}/features/{gfid}/links")
@Tag(name = "Weblinks")
public class WeblinkController {

    CommentOrLinkService commentService = new CommentOrLinkService();

    @GET
    @Path("/")
    @Produces({ MediaType.APPLICATION_JSON })
    @Operation(summary = "Retrieve all web links of a feature")
    public Response getLayerWeblinks(
            @Context SecurityContext sec,
            @PathParam("workspace") String workspace,
            @PathParam("layername") String dataLayerName,
            @PathParam("gfid") int gfid) throws IOException, SQLException {

        try {
            User user = UserRepository.getUser(sec.getUserPrincipal().getName()).get();

            // If user has no write permission on that workspace, he would already have been
            // thrown out by authorisation layer
            int layerId = DatasetRepository.getLayerId(workspace, dataLayerName);

            List<Weblink> weblinks = commentService.getWeblinks(user, workspace, layerId, gfid);
            return Response.ok().entity(weblinks).build();

        } catch (SQLException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }

    @DELETE
    @Path("/{linkId}")
    @Operation(summary = "Delete a web link from a feature of a layer")
    public Response deleteFeatureComment(
            @Context SecurityContext sec,
            @PathParam("workspace") String workspace,
            @PathParam("layername") String dataLayerName,
            @PathParam("gfid") int gfid,
            @PathParam("linkId") int linkId) throws SQLException {

        User user = UserRepository.getUser(sec.getUserPrincipal().getName()).get();
        AdminTools.checkAdminPermission(user.getId(), workspace);

        int layerId = DatasetRepository.getLayerId(workspace, dataLayerName);
        commentService.deleteCommentOrWeblink(layerId, gfid, linkId);

        return Response.status(200).build();
    }
}
