package de.narimo.georepo.server.api.documents;

import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.sql.SQLException;
import java.util.Arrays;
import java.util.List;

import javax.imageio.IIOException;
import javax.imageio.ImageIO;
import javax.servlet.ServletContext;
import javax.ws.rs.NotFoundException;

import de.narimo.commons.dto.geometa.User;
import de.narimo.georepo.server.GeorepoTools;
import de.narimo.georepo.server.repository.DatasetRepository;
import de.narimo.georepo.server.repository.ImageRepository;
import de.narimo.georepo.server.tools.TableTools;

public class ImageAndDocumentService {

    List<String> allowedImageFileTypes = Arrays.asList("image/png", "image/jpg", "image/jpeg", "image/gif");
    List<String> allowedDocumentFileTypes = Arrays.asList("application/pdf");

    public List<String> getFeatureImages(int layerId, int gfid)
            throws IOException, SQLException {

        // if no image table exists, return empty list
        boolean tableExists = ImageRepository.checkImageTableExists(layerId);
        if (!tableExists) {
            System.out.println("Image and document table does not yet exist for layer " + layerId
                    + ". No images or documents present.");
            return Arrays.asList();
        }

        List<String> images = ImageRepository.getFeatureImages(layerId, gfid);

        for (String image : images) {
            if (getFileExtension(image) == null) {
                System.out.println("Invalid file name entry for " + layerId + " and file " + image + ".");
            }
        }

        return images;
    }

    public List<FeatureDocument> getFeatureImagesOrDocuments(User user, String workspace, int layerId, int gfid,
            DocumentType documentType)
            throws IOException, SQLException {

        // if no image table exists, return empty list
        boolean tableExists = ImageRepository.checkImageTableExists(layerId);
        if (!tableExists) {
            System.out.println("Image and document table does not yet exist for layer " + layerId
                    + ". No images or documents present.");
            return Arrays.asList();
        }

        return ImageRepository.getFeatureImagesOrDocuemnts(user, workspace, layerId, gfid, documentType);

//        for (String image : images) {
//            if (getFileExtension(image) == null) {
//                System.out.println("Invalid file name entry for " + layerId + " and file " + image + ".");
//            }
//        }
    }

    public byte[] getFeatureImageOrDocument(ServletContext ctx, String workspace, int layerId, int gfid,
            String filename,
            DocumentType documentType)
            throws IOException, IIOException, SQLException {

        String fileExtension = getFileExtension(filename);

        if (!isAllowedFileType(getFileExtension(filename), documentType)) {
            throw new InternalError("The file has an incorrect file type.");
        }

        if (!ImageRepository.isImageFromFeature(layerId, gfid, filename)) {
            throw new NotFoundException();
        }

        String fileBasePath = ctx.getInitParameter("WORKSPACE_IMAGES_BASE_PATH");
        if (fileBasePath == null) {
            throw new RuntimeException("File base path is not set.");
        }

        String filePath = fileBasePath + "/" + workspace + "/images/" + layerId + "/" + filename;
        System.out.println("Looking up image or document: " + workspace + "/images/" + layerId + "/" + filename);

        if (documentType == DocumentType.IMAGE) {
            BufferedImage image = ImageIO.read(new File(filePath));
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            ImageIO.write(image, fileExtension, baos);
            return baos.toByteArray();
        } else if (documentType == DocumentType.DOCUMENT) {
            return Files.readAllBytes(new File(filePath).toPath());
        }
        throw new NotFoundException();
    }

    public int addImageOrDocument(ServletContext ctx, User user, String workspace, InputStream fileInputStream,
            String filename, int layerId, int gfid, DocumentType documentType)
            throws IOException, SQLException {

        try {
            if (!DatasetRepository.doesFeatureExist(layerId, gfid)) {
                throw new NotFoundException();
            }

            String fileExtension = getFileExtension(filename);

            // do checks on image
            if (!isAllowedFileType(fileExtension, documentType)) {
                if (documentType == DocumentType.IMAGE) {
                    throw new IOException("Unsupported filetype " + fileExtension
                            + ". Must be image/png, image/jpg, image/jpeg or image/gif.");
                }
                throw new IOException("Unsupported filetype " + fileExtension
                        + ". Must be application/pdf.");
            }

            // get the images base path
            String fileBasePath = ctx.getInitParameter("WORKSPACE_IMAGES_BASE_PATH");
            if (fileBasePath == null) {
                throw new RuntimeException("Image upload base path is not set.");
            }

            // create the base directory, if not exists
            File basePath = new File(fileBasePath);
            if (basePath.exists() && !basePath.isDirectory()) {
                throw new RuntimeException("Workspace directory must be a directory.");
            }

            // add the image metadata to database and get the new image id
            boolean imgTableExists = ImageRepository.checkImageTableExists(layerId);
            if (!imgTableExists) {
                System.out.println("Image table does not yet exist for layer " + layerId + ". Creating it...");
                ImageRepository.createImageTable(layerId);
            }

            boolean imgDiffTableExists = ImageRepository.checkImageDiffTableExists(layerId);
            if (!imgDiffTableExists) {
                System.out.println("Diff table does not yet exist for layer " + layerId + ". Creating it...");
                ImageRepository.createImageDiffTable(layerId);
            }

            String serverGeneratedFilename = GeorepoTools.createAlphanumericString(12) + "."
                    + fileExtension.toLowerCase();

            // add the image to the filesystem
            String imgDirPath = basePath + "/" + workspace + "/images/" + layerId + "/upload";
            File imagePath = new File(imgDirPath);
            if (!imagePath.exists()) {
                imagePath.mkdirs();
            }

            File outFile = new File(imagePath + "/" + serverGeneratedFilename);
            System.out.println("Filepath for new file: " + outFile.toString());

            if (documentType == DocumentType.IMAGE) {
                BufferedImage bufferedImage = ImageIO.read(fileInputStream);
                if (bufferedImage.getWidth() > 1920 || bufferedImage.getHeight() > 1080) {
                    throw new IOException("Image dimensions should not exceed 1920x1080 pixels.");
                }
                ImageIO.write(bufferedImage, fileExtension, outFile);
            } else if (documentType == DocumentType.DOCUMENT) {
                Files.copy(
                        fileInputStream,
                        outFile.toPath(),
                        StandardCopyOption.REPLACE_EXISTING);

                if (outFile.length() > 10240000) {
                    Files.delete(outFile.toPath());
                    throw new IOException("File size should not exceed 10MB.");
                }
            }

            int diffImageId = ImageRepository.insertUnreviewedImage(layerId, user.getId(), gfid,
                    serverGeneratedFilename, documentType);

//        inform the admins that a new image has been uploaded to the workspace

//        boolean notifyAdmin = true;
//        if (notifyAdmin) {
//            Notifier.notifyWorkspaceAdminsAboutPOIChanges(ctx, dataWorkspace, dataLayerName);
//        }

            return diffImageId;

        } finally {
            if (fileInputStream != null) {
                fileInputStream.close();
            }
        }
    }

    public void acceptImageOrDocument(ServletContext ctx, int layerId, int gfid, int diffImageId, String workspace,
            DocumentType documentType)
            throws IOException {

        String imagesTable = TableTools.getImageTableName(layerId);
        String diffImagesTable = TableTools.getImageDiffTableName(layerId);

        String filename = ImageRepository.acceptNewImageOrDocument(imagesTable, diffImagesTable, gfid, diffImageId);
        if (filename == null) {
            throw new RuntimeException("Invalid contents in diffImageTable. Filename is null.");
        }

        String imgBasePath = ctx.getInitParameter("WORKSPACE_IMAGES_BASE_PATH");
        if (imgBasePath == null) {
            throw new RuntimeException("Image upload base path is not set.");
        }

        String reviewedImageDirPath = imgBasePath + "/" + workspace + "/images/" + layerId;

        String imgDirPath = reviewedImageDirPath + "/upload";
        File imagePath = new File(imgDirPath + "/" + filename);

        if (!imagePath.exists()) {
            throw new RuntimeException("Image upload directory does not exist.");
        }

        File reviewedImageDir = new File(reviewedImageDirPath);
        if (!reviewedImageDir.exists()) {
            reviewedImageDir.mkdirs();
        }

        File outFile = new File(reviewedImageDir + "/" + filename);

        if (documentType == DocumentType.IMAGE) {
            BufferedImage bufferedImage = ImageIO.read(imagePath);
            ImageIO.write(bufferedImage, getFileExtension(filename), outFile);
        } else if (documentType == DocumentType.DOCUMENT) {
            Files.move(imagePath.toPath(), outFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
        }
        return;
    }

    public String getFileExtension(String filename) {
        if (filename == null) {
            return null;
        }
        return GeorepoTools.getFileExtension(filename);
    }

    /**
     * Checks a simple file type like png, jpg for being valid.
     * 
     * @return
     */
    private boolean isAllowedFileType(String fileType, DocumentType documentType) {
        if (fileType == null)
            return false;
        if (documentType == DocumentType.IMAGE && allowedImageFileTypes.contains("image/" + fileType.toLowerCase())) {
            return true;
        }
        if (documentType == DocumentType.DOCUMENT
                && allowedDocumentFileTypes.contains("application/" + fileType.toLowerCase())) {
            return true;
        }
        return false;
    }
}
