package de.narimo.georepo.server.api.noauth;

import java.io.IOException;
import java.io.InputStream;

import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.GET;
import javax.ws.rs.NotFoundException;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.ext.Provider;

import com.fasterxml.jackson.databind.ObjectMapper;

import de.narimo.geocore.ws.registration.RegistrationProvider;
import de.narimo.geocore.ws.registration.UserRegistrationDetails;
import de.narimo.georepo.server.notification.Notifier;
import io.swagger.v3.oas.annotations.tags.Tag;

@Provider
@Path("")
@Tag(name = "RegistrationController")
public class RegistrationController {

    @POST
    @Path("/register")
    public static Response registerNewUser(
            @Context ServletContext ctx,
            @Context HttpServletRequest request,
            InputStream body) {

        try {
            RegistrationCheck.doCrawlerCheck(request.getHeader("user-agent"));

            ObjectMapper mapper = new ObjectMapper();
            UserRegistrationDetails registrationDetails = mapper.readValue(body, UserRegistrationDetails.class);

            RegistrationCheck.doEmailCheck(registrationDetails.getEmail());
            String secret = RegistrationProvider.registerUser(registrationDetails);

            String georepoPlatform = ctx.getInitParameter("GEOREPO_URL_EXT");
            String registrationLink = georepoPlatform + "/georepo-server/registration/confirm?"
                    + "m=" + registrationDetails.getEmail() + "&s=" + secret;

            Notifier.notifyUserAboutRegistrationConfirmationLink(ctx, registrationDetails, registrationLink);
            Notifier.notifyNarimoAboutRegistration(ctx, registrationDetails);

            return Response.ok().entity("Registration successful but needs confirmation.").build();

        } catch (IOException e) {
            e.printStackTrace();
            if (e.getMessage().contains("Username or email is already registered")) {
                return Response.status(Status.CONFLICT).entity("The email is already registered.").build();
            }
            return Response.status(Status.BAD_REQUEST).build();
        } catch (NotFoundException e) {
            return Response.status(Status.NOT_FOUND).build();
        } catch (InternalError | RuntimeException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }

    @GET
    @Path("/confirm")
    public static Response confirmNewUser(
            @Context ServletContext ctx,
            @Context HttpServletRequest request,
            @QueryParam("m") String email,
            @QueryParam("s") String secret) {

        try {
            RegistrationCheck.doCrawlerCheck(request.getHeader("user-agent"));

            if (email == null || secret == null) {
                throw new IOException("Invalid confirmation link.");
            }

            RegistrationProvider.confirmRegistration(email, secret);

            String confirmation = "<html><center><div style='color:#224366;margin:40px'>Your registration has been successfully confirmed.</div></center></html>";
            return Response.ok().entity(confirmation).build();

        } catch (IOException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        } catch (NotFoundException e) {
            return Response.status(Status.NOT_FOUND).build();
        } catch (InternalError | RuntimeException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }
}
