package de.narimo.georepo.server.repository;

import java.io.IOException;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.ws.rs.NotFoundException;

import org.geotools.feature.FeatureCollection;

import de.narimo.commons.jdbc.JDBCConnectionJNDI;
import de.narimo.georepo.server.GeorepoConstants;
import de.narimo.georepo.server.tools.QueryCheck;
import de.narimo.georepo.server.tools.TableTools;

public class DatasetRepository extends ParentDatasetRepository {

    public static FeatureCollection getDatasets(int layerId, long userId) throws SQLException {

        String dataTableName = TableTools.getDataTableName(layerId);
        QueryCheck.checkTableName(dataTableName);

        Map<String, String> columnsAndTypes = getColumnTypes(dataTableName,
                GeorepoConstants.DEFAULTSCHEMA);

        Map<String, Class> propertyTypes = new HashMap<String, Class>();

        for (String colName : columnsAndTypes.keySet()) {
            String colType = columnsAndTypes.get(colName);
            if (colType.equals("integer")) {
                propertyTypes.put(colName, Integer.class);
            } else if (colType.equals("double precision")) {
                propertyTypes.put(colName, Double.class);
            } else {
                propertyTypes.put(colName, String.class);
            }
        }

        Map<Integer, List<String>> tags = null;
        if (checkTagTableExists(layerId)) {
            if (propertyTypes.containsKey("tags")) {
                System.out.println("Warning: Data table already contains a property 'tags'. "
                        + "Cannot match tags from tags table to the dataset.");
            } else {
                columnsAndTypes.put("tags", "text");
                propertyTypes.put("tags", String[].class);
                tags = getTagsByGfid(layerId);
            }
        }

        String query = "SELECT ST_AsText(the_geom), * from public." + dataTableName + ";";

        return createSimpleFeatures(dataTableName, propertyTypes, columnsAndTypes, query, tags);
    }

    public static int getLayerId(String workspace, String featureType) {

        JDBCConnectionJNDI jdbcMeta = null;

        try {
            jdbcMeta = new JDBCConnectionJNDI("jdbc/georepoMetaResource");

            String sql = "SELECT id from public.datasets WHERE workspace = ? AND featuretype = ?;";
            System.out.println(sql);

            PreparedStatement ps = jdbcMeta.prepareStatement(sql);
            ps.setString(1, workspace);
            ps.setString(2, featureType);

            ResultSet rs = ps.executeQuery();

            int count = 0;
            Integer id = null;
            while (rs.next()) {
                id = rs.getInt("id");
                count++;
            }

            if (count > 1) {
                throw new NotFoundException(
                        "Could not determine distinct layer id. Please review datasets table consistency for workspace "
                                + workspace + " and feature type " + featureType + "!");
            }

            return id;
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (jdbcMeta != null) {
                jdbcMeta.closeAll();
                jdbcMeta = null;
            }
        }
        throw new NotFoundException("Could not determine layer id.");
    }

    /**
     * Checks, whether a feature with gfid exists for the given layer.
     * 
     * @param layerId
     * @param gfid
     * @return
     * @throws SQLException
     * @throws IOException
     */
    public static boolean doesFeatureExist(int layerId, int gfid)
            throws SQLException {

        JDBCConnectionJNDI jdbcData = null;

        String dataTable = TableTools.getDataTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql = "SELECT count(*) FROM " + dataTable + " WHERE gfid = ?;";
            System.out.println(sql);

            PreparedStatement ps = jdbcData.prepareStatement(sql);
            ps.setInt(1, gfid);
            ResultSet rs = ps.executeQuery();

            rs.next();
            int count = rs.getInt("count");
            if (count > 0) {
                return true;
            }

        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
        return false;
    }

    public static List<String> getCategories(int layerId, String optCategoryColumn) throws SQLException {

        JDBCConnectionJNDI jdbcData = null;

        String dataTable = TableTools.getDataTableName(layerId);

        List<String> categories = new ArrayList<>();

        String categoryColumn = "category";
        if (optCategoryColumn != null && optCategoryColumn.chars().allMatch(Character::isLetter)) {
            categoryColumn = optCategoryColumn;
        }

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql0 = "SELECT column_name FROM information_schema.columns WHERE table_name=? and column_name=?;";
            PreparedStatement ps0 = jdbcData.prepareStatement(sql0);
            ps0.setString(1, dataTable);
            ps0.setString(2, categoryColumn);
            ResultSet rs0 = ps0.executeQuery();
            if (!rs0.next()) {
                return new ArrayList<>();
            }

            String sql = "SELECT DISTINCT " + categoryColumn + " FROM " + dataTable;
            System.out.println(sql);

            PreparedStatement ps = jdbcData.prepareStatement(sql);
            ResultSet rs = ps.executeQuery();

            while (rs.next()) {
                String category = rs.getString(1);
                if (category != null) {
                    categories.add(rs.getString(1));
                }
            }

        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
        return categories;
    }

    public static List<String> getTags(int layerId) throws SQLException {

        JDBCConnectionJNDI jdbcData = null;

        String tagTable = TableTools.getTagTableName(layerId);

        List<String> tags = new ArrayList<>();

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql = "SELECT DISTINCT tag FROM " + tagTable;
            System.out.println(sql);

            PreparedStatement ps = jdbcData.prepareStatement(sql);
            ResultSet rs = ps.executeQuery();

            while (rs.next()) {
                String tag = rs.getString(1);
                if (tag != null) {
                    tags.add(rs.getString(1));
                }
            }

        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
        return tags;
    }

    public static Map<Integer, List<String>> getTagsByGfid(int layerId) throws SQLException {

        JDBCConnectionJNDI jdbcData = null;

        String tagTable = TableTools.getTagTableName(layerId);

        Map<Integer, List<String>> tags = new HashMap<>();

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql = "SELECT DISTINCT gfid, tag FROM " + tagTable + " ORDER BY gfid, tag;";
            System.out.println(sql);

            PreparedStatement ps = jdbcData.prepareStatement(sql);
            ResultSet rs = ps.executeQuery();

            while (rs.next()) {
                String tag = rs.getString(2);
                if (tag != null) {
                    int gfid = rs.getInt(1);
                    if (gfid != 0) {
                        List<String> tagList = tags.get(gfid);
                        if (tagList == null) {
                            tagList = new ArrayList<>();
                        }
                        tagList.add(tag);
                        tags.put(gfid, tagList);
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
        return tags;
    }

    public static boolean checkTagTableExists(int layerId) throws SQLException {

        JDBCConnectionJNDI jdbcData = null;

        String tagTable = TableTools.getTagTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql = "SELECT 1 FROM " + tagTable + ";";
            System.out.println(sql);

            jdbcData.executeQuery(sql);

            return true;
        } catch (Exception e) {
            if (e.getMessage().contains("does not exist")) {
                return false;
            }
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    public static void createTagTable(int layerId) throws SQLException {
        JDBCConnectionJNDI jdbcData = null;

        String tagTable = TableTools.getTagTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql = "CREATE TABLE public." + tagTable + "("
                    + "id SERIAL PRIMARY KEY,"
                    + "gfid integer NOT NULL,"
                    + "tag text NOT NULL"
                    + ");";

            System.out.println(sql);
            jdbcData.execute(sql);

        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }
}
