package de.narimo.georepo.server.repository;

import java.io.IOException;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import de.narimo.commons.dto.geometa.User;
import de.narimo.commons.jdbc.JDBCConnectionJNDI;
import de.narimo.geocore.ws.repository.UserRepository;
import de.narimo.georepo.server.GeorepoConstants;
import de.narimo.georepo.server.api.documents.DocumentType;
import de.narimo.georepo.server.api.documents.FeatureDocument;
import de.narimo.georepo.server.tools.AdminTools;
import de.narimo.georepo.server.tools.QueryCheck;
import de.narimo.georepo.server.tools.TableTools;

public class ImageRepository {

    /**
     * Creates a table that stores images related to a POI of the layer that this
     * table is related to.
     * 
     * @param layerId
     * @throws SQLException
     * @throws IOException
     */
    public static void createImageTable(int layerId) throws SQLException, IOException {
        JDBCConnectionJNDI jdbcData = null;

        String imgDataTable = TableTools.getImageTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql = "CREATE TABLE public." + imgDataTable + "("
                    + "id SERIAL PRIMARY KEY,"
                    + "gfid integer NOT NULL,"
                    + "filename text UNIQUE NOT NULL,"
                    + "description text,"
                    + "type text NOT NULL,"
                    + "insertedby integer NOT NULL,"
                    + "insertedat timestamp without time zone DEFAULT timezone('utc'::text, now()) NOT NULL"
                    + ");";

            System.out.println(sql);
            jdbcData.execute(sql);

        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    /**
     * Creates an image diff table from the structure of an image table. And adds a
     * grpstatus column.
     *
     * @param dataTable
     */
    public static void createImageDiffTable(int layerId) throws SQLException, IOException {
        JDBCConnectionJNDI jdbcData = null;

        String imgDataTable = TableTools.getImageTableName(layerId);
        String imgDiffTable = TableTools.getImageDiffTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql = "CREATE TABLE public." + imgDiffTable + ""
                    + " (LIKE " + imgDataTable
                    + " INCLUDING DEFAULTS INCLUDING CONSTRAINTS INCLUDING INDEXES);"
                    + " ALTER TABLE " + imgDiffTable
                    + " ADD COLUMN " + GeorepoConstants.GRPSTATUSCOLUMN + " character varying(1) NOT NULL;";

            System.out.println(sql);
            jdbcData.execute(sql);

        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    /**
     * Checks, whether an image table exists for a given layer id.
     * 
     * @param layerId
     * @return
     * @throws SQLException
     * @throws IOException
     */
    public static boolean checkImageTableExists(int layerId) throws SQLException, IOException {

        JDBCConnectionJNDI jdbcData = null;

        String imgTable = TableTools.getImageTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql = "SELECT 1 FROM " + imgTable + ";";
            System.out.println(sql);

            jdbcData.executeQuery(sql);

            return true;
        } catch (Exception e) {
            if (e.getMessage().contains("does not exist")) {
                return false;
            }
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    /**
     * Checks, if a image diff table exist for a given layer id.
     * 
     * @param layerId
     * @return
     * @throws SQLException
     * @throws IOException
     */
    public static boolean checkImageDiffTableExists(int layerId) throws SQLException, IOException {

        JDBCConnectionJNDI jdbcData = null;

        String imgDiffTable = TableTools.getImageDiffTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql = "SELECT 1 FROM " + imgDiffTable + ";";
            System.out.println(sql);

            jdbcData.executeQuery(sql);

            return true;
        } catch (Exception e) {
            e.printStackTrace();
            if (e.getMessage().contains("does not exist")) {
                return false;
            }
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    /**
     * Inserts a new image and returns its id.
     * 
     * @param layerId
     * @param userId
     * @param gfid
     * @param generatedFilename
     * @return
     * @throws SQLException
     * @throws IOException
     */
    public static int insertUnreviewedImage(int layerId, Integer userId, int gfid, String generatedFilename,
            DocumentType documentType)
            throws SQLException, IOException {

        if (userId == null) {
            throw new RuntimeException();
        }

        JDBCConnectionJNDI jdbcData = null;

        String imgDiffTable = TableTools.getImageDiffTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql = "INSERT INTO " + imgDiffTable
                    + " (gfid, filename, type, insertedby, grpstatus) VALUES (?, ?, ?, ?, ?) RETURNING id;";
            System.out.println(sql);

            PreparedStatement ps = jdbcData.prepareStatement(sql);
            ps.setInt(1, gfid);
            ps.setString(2, generatedFilename);
            ps.setString(3, documentType.toString());
            ps.setInt(4, userId);
            ps.setString(5, "n");

            ResultSet rs = ps.executeQuery();

            while (rs.next()) {
                return rs.getInt("id");
            }
            throw new RuntimeException("Could not insert image.");

        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    /**
     * Retrieves a list of image names for a feature.
     * 
     * @param layerId
     * @param gfid
     * @return
     * @throws SQLException
     * @throws IOException
     */
    public static List<String> getFeatureImages(int layerId, int gfid)
            throws SQLException, IOException {

        JDBCConnectionJNDI jdbcData = null;

        String imgTable = TableTools.getImageTableName(layerId);
        List<String> imageOrDocumentNames = new ArrayList<String>();

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql = "SELECT filename FROM " + imgTable + " WHERE gfid = ? "
                    + "AND type = 'IMAGE';";
            System.out.println(sql);

            PreparedStatement ps = jdbcData.prepareStatement(sql);
            ps.setInt(1, gfid);
            ResultSet rs = ps.executeQuery();

            while (rs.next()) {
                String filename = rs.getString("filename");
                if (filename == null) {
                    throw new InternalError("Invalid file entry for layer=" + layerId + " and gfid=" + gfid + ".");
                }
                imageOrDocumentNames.add(filename);
            }

        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
        return imageOrDocumentNames;
    }

    /**
     * Retrieves a list of documents with related information for each feature.
     * 
     * @param layerId
     * @param gfid
     * @param documentType
     * @return
     * @throws SQLException
     * @throws IOException
     */
    public static List<FeatureDocument> getFeatureImagesOrDocuemnts(User requestingUser, String workspace, int layerId,
            int gfid, DocumentType documentType)
            throws SQLException {

        JDBCConnectionJNDI jdbcData = null;

        String imageOrDocumentsTable = TableTools.getImageTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql = "SELECT filename, description, insertedat, insertedby FROM " + imageOrDocumentsTable
                    + " WHERE gfid = ? AND type = '" + documentType.toString() + "';";
            System.out.println(sql);

            PreparedStatement ps = jdbcData.prepareStatement(sql);
            ps.setInt(1, gfid);
            ResultSet rs = ps.executeQuery();

            List<FeatureDocument> documents = new ArrayList<>();
            List<Integer> creators = new ArrayList<>();
            while (rs.next()) {
                String filename = rs.getString("filename");
                String insertedat = rs.getString("insertedat");
                String description = rs.getString("description");
                int insertedby = rs.getInt("insertedby");
                FeatureDocument document = new FeatureDocument();
                document.setName(filename);
                document.setCreatedat(insertedat);
                document.setDescription(description);
                // this is the user id but needs to be transformed to a user name
                document.setCreator(String.valueOf(insertedby));

                creators.add(insertedby);

                documents.add(document);
            }

            List<User> users = UserRepository.getUsersById(creators);
            Map<Integer, User> userMap = new HashMap<Integer, User>();
            for (User user : users) {
                userMap.put(user.getId(), user);
            }

            for (FeatureDocument document : documents) {
                int userid = Integer.valueOf(document.getCreator());
                String documentCreatorMail = userMap.get(userid).getEmail();

                document.setCreator(documentCreatorMail.substring(0, documentCreatorMail.indexOf('@')));

                if (AdminTools.isWorkspaceAdmin(requestingUser.getId(), workspace)) {
                    document.setCreatormail(documentCreatorMail);
                }
            }

            return documents;
        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    /**
     * Accept an image from an images diff table.
     *
     * @param imagesTable
     * @param diffImagesTable
     * @param gfid
     * @param diffImageId
     * @throws IOException
     */
    public static String acceptNewImageOrDocument(String imagesTable, String diffImagesTable,
            int gfid, int diffImageId) throws IOException {

        QueryCheck.checkTableName(imagesTable);
        QueryCheck.checkDiffTableName(diffImagesTable);

        JDBCConnectionJNDI jdbcData = null;

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql = "SELECT filename FROM " + diffImagesTable + " WHERE gfid = " + gfid
                    + " AND id = " + diffImageId + " AND grpstatus = 'n';";
            ResultSet rs = jdbcData.executeQuery(sql);
            rs.next();
            String filename = rs.getString("filename");

            String accept = "INSERT INTO " + imagesTable + " (filename,type,gfid,insertedby,insertedat) "
                    + "SELECT filename,type,gfid,insertedby,insertedat from " + diffImagesTable + " WHERE gfid = "
                    + gfid + " AND id = " + diffImageId + " AND grpstatus = 'n';";

            accept += "DELETE FROM " + diffImagesTable + " WHERE gfid = " + gfid + " AND id = " + diffImageId
                    + " AND grpstatus = 'n';";

            System.out.println(accept);

            jdbcData.execute(accept);

            return filename;

        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    /**
     * Check, if there is an image with the given name for the given layer and
     * feature.
     * 
     * @param layerId
     * @param gfid
     * @param filename
     * @return
     * @throws SQLException
     * @throws IOException
     */
    public static boolean isImageFromFeature(int layerId, int gfid, String filename)
            throws SQLException, IOException {

        JDBCConnectionJNDI jdbcData = null;

        String imgTable = TableTools.getImageTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql = "SELECT count(*) FROM " + imgTable + " WHERE gfid = ? AND filename = ?;";
            System.out.println(sql);

            PreparedStatement ps = jdbcData.prepareStatement(sql);
            ps.setInt(1, gfid);
            ps.setString(2, filename);
            ResultSet rs = ps.executeQuery();

            rs.next();
            int count = rs.getInt("count");
            if (count == 1) {
                return true;
            }

        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
        return false;
    }

}
