package de.narimo.georepo.server.repository;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.geotools.feature.FeatureCollection;
import org.locationtech.jts.geom.Geometry;
import org.opengis.feature.simple.SimpleFeature;

import de.narimo.commons.jdbc.JDBCConnectionJNDI;
import de.narimo.georepo.server.GeorepoConstants;
import de.narimo.georepo.server.layer.FeatureBuilder;
import de.narimo.georepo.server.layer.GeometryType;
import de.narimo.georepo.server.tools.GeometryTools;
import de.narimo.georepo.server.tools.QueryCheck;

public class ParentDatasetRepository {

    static Map<String, String> getColumnTypes(String tableName, String schemaName) {
        JDBCConnectionJNDI jdbcData = null;

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql0 = "SELECT column_name, data_type FROM information_schema.columns " + "WHERE table_schema = ? "
                    + "AND table_name   = ?;";

            PreparedStatement ps0 = jdbcData.prepareStatement(sql0);
            ps0.setString(1, schemaName);
            ps0.setString(2, tableName);

            List<String> reservedColumnNames = new ArrayList<>();
            reservedColumnNames.add(GeorepoConstants.GRPFIDCOLUMN);
            // reservedColumnNames.add(GeorepoConstants.GFIDCOLUMN);
            reservedColumnNames.add(GeorepoConstants.GEOMETRYCOLUMN);
            reservedColumnNames.add(GeorepoConstants.GRPSTATUSCOLUMN);

            Map<String, String> columnNames = new HashMap<>();

            ResultSet rs0 = ps0.executeQuery();
            while (rs0.next()) {
                String columnName = rs0.getString("column_name");
                if (!reservedColumnNames.contains(columnName)) {
                    columnNames.put(columnName, rs0.getString("data_type"));
                }
            }
            return columnNames;
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
        return null;
    }

    public static FeatureCollection createSimpleFeatures(String dataTableName, Map<String, Class> propertyTypes,
            Map<String, String> columnsAndTypes, String query, Map<Integer, List<String>> tags) {

        JDBCConnectionJNDI jdbcData = null;

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            GeometryType geometryType = detectGeometryType(dataTableName);
            if (geometryType == null) {
                System.out.println("Detected geometry type is: " + geometryType);
            }

            FeatureBuilder fb = new FeatureBuilder(geometryType, propertyTypes);
            List<SimpleFeature> simpleFeatures = new ArrayList<>();

            ResultSet rs = jdbcData.executeQuery(query);

            String wkt = null;
            while (rs.next()) {
                int gfid = rs.getInt("gfid");
                wkt = rs.getString("st_astext");

                Geometry geometry = null;
                if (wkt != null) {
                    geometry = GeometryTools.createGeometry(wkt);
                }

                Map<String, Object> properties = new HashMap<>();
                for (String colName : columnsAndTypes.keySet()) {
                    if (colName.equals("tags") && tags != null) {
                        if (tags.get(gfid) != null && tags.get(gfid).size() > 0) {
                            properties.put(colName, tags.get(gfid));
                        }
                    } else {
                        properties.put(colName, rs.getString(colName));
                    }
                }

                SimpleFeature simpleFeature = fb.createSimpleFeature(geometry, properties);
                simpleFeatures.add(simpleFeature);
            }
            return fb.createFeatureCollection(simpleFeatures);

        } catch (SQLException e) {
            e.printStackTrace();
        } catch (Exception e) {
            throw new RuntimeException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
        return null;
    }

    /**
     * Intends to detect the geometry type from the the_geom colum of the first
     * entry of the data table.
     * 
     * @param dataTableName
     * @return
     * @throws Exception
     */
    public static GeometryType detectGeometryType(String dataTableName) {
        QueryCheck.checkTableName(dataTableName);

        JDBCConnectionJNDI jdbcData = null;

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String s = "SELECT ST_GeometryType(ST_AsText(the_geom)) from public." + dataTableName
                    + " WHERE the_geom IS NOT NULL limit 1;";
            System.out.println(s);

            ResultSet rs = jdbcData.executeQuery(s);

            if (rs.next()) {
                String geomType = rs.getString("st_geometrytype");
                if (geomType == null) {
                    return null;
                } else if (geomType.equals("ST_Point")) {
                    return GeometryType.POINT;
                } else if (geomType.equals("ST_LineString")) {
                    return GeometryType.LINESTRING;
                } else if (geomType.equals("ST_Polygon")) {
                    return GeometryType.POLYGON;
                }
            }
            return null;

        } catch (Exception e) {
            throw new RuntimeException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }
}
