package de.narimo.georepo.server.api;

import java.util.List;

import javax.servlet.ServletContext;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.SecurityContext;
import javax.ws.rs.ext.Provider;

import de.narimo.commons.json.JsonConverter;
import de.narimo.georepo.server.GeorepoConstants;
import de.narimo.georepo.server.GeorepoDatasource;
import de.narimo.georepo.server.GeorepoTools;
import de.narimo.georepo.server.dto.FeatureType;
import de.narimo.georepo.server.geoserver.GeoserverLayer;
import de.narimo.georepo.server.geoserver.GeoserverWorkspace;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;

/**
 * georepo.com layer API
 *
 * @author Ulrich Mann
 *
 */
@Provider
@Path("/workspaces/{workspace}/layers")
@Tag(name = "Layers")
public class LayerController {

    @GET
    @Path("/")
    @Produces(MediaType.APPLICATION_JSON)
    @Operation(description = "Retrieve workspace layers")
    public static Response getWorkspaceLayers(
            @Context SecurityContext sec,
            @Context ServletContext ctx,
            @PathParam("workspace") String workspace) {

        try {
            GeoserverLayer layer = new GeoserverLayer(ctx);
            GeoserverWorkspace gworkspace = new GeoserverWorkspace(layer.getGeoserverRestUrl(), layer.getHeaders(),
                    layer.getGeoserverUser(), layer.getGeoserverPass());

            List<FeatureType> layers = gworkspace.getWorkspaceFeatureTypes(workspace, false);

            return Response.ok().entity(JsonConverter.toJson(layers)).build();
        } catch (Exception e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }

    /**
     * Prerequisites for csv import: - only point data supported yet - column lon
     * for longitude coordinates - column lat for latitude coordinates - filename
     * must only contain alphanumerics, dashes and underscores - filename must end
     * .csv (postgres insert function restriction) - only allowed csvdelimiter is
     * semicolon (;) - currently only utf8 charset allowed, i.e., no umlauts, ß,
     * etc. in csv content - numeric values in csv are inserted as text to database
     *
     * TODO: check: do NOT allow feature type names containing whitespaces! This
     * will mess up queries that are not correctly url encoded or even then.
     *
     * NB: Conventions for new layers, datasources and workspaces: - datasource
     * starts with prefix gd_ and concatenates workspace name - layer name (not
     * layer title!) must not contain whitespaces and end with _gd_ plus the newly
     * created layerId
     *
     * @param sec
     * @param ctxt
     * @param workspace
     * @param sridAsString
     * @param delimiter
     * @param fileName
     * @param readableLayerName
     * @param fileType
     * @param optColumnTypes
     * @return
     * @throws Exception
     */
    @POST
    @Path("")
    @Operation(summary = "Create a new layer", description = "The CSV resource to create the layer from must already reside on the server. "
            + "It is preferred to use the import methods which will also handle creating a layer from an uploaded resource.")
    public static Response createLayer(
            @Context ServletContext ctx,

            @PathParam("workspace") String workspace,

            @QueryParam("srid") String sridAsString, // EPSG code, defaults to
                                                     // 4326
            @QueryParam("delimiter") String delimiter, // necessary for geocsv,
                                                       // defaults to ;
            @QueryParam("filename") String fileName, // should be the name as
                                                     // how it lives in
                                                     // geoserver_data_dir
            @QueryParam("layername") String readableLayerName,
            @QueryParam("filetype") String fileType,
            @QueryParam("columntypes") String optColumnHeaders, // only, if no fileType provided (empty layer import)
            @QueryParam("columntypes") String optColumnTypes // optional
                                                             // comma-separated
                                                             // string with
                                                             // postgres column
                                                             // types, e.g.
                                                             // real,text,text,integer...
    ) throws Exception {

        System.out.println("Column types to use: " + optColumnTypes);

        String[] columnTypes = optColumnTypes == null ? null : optColumnTypes.split("\\s*,\\s*");
        String[] columnHeaders = optColumnHeaders == null ? null : optColumnHeaders.split("\\s*,\\s*");

        if (fileType == null) {
            return GeorepoTools.importEmptyLayer(ctx, sridAsString, readableLayerName, workspace, columnHeaders,
                    columnTypes);
        }
        if (fileType.equals("geocsv")) {
            return GeorepoTools.importGeoCSV(
                    ctx, ctx.getInitParameter(GeorepoConstants.geoserverDataDir), sridAsString, delimiter, fileName,
                    readableLayerName, workspace, columnTypes);
        } else {
            throw new IllegalArgumentException("Parameter filetype must specify a supported filetype.");
        }

    }

    @PUT
    @Path("/{layername}")
    @Operation(summary = "Update an existing layer")
    public static Response updateLayerContent(
            @Context ServletContext ctx,

            @PathParam("workspace") String workspace,
            @PathParam("layername") String layerName,

            @QueryParam("srid") String sridAsString, // EPSG code, defaults to
                                                     // 4326
            @QueryParam("delimiter") String delimiter, // necessary for geocsv,
                                                       // defaults to ;
            @QueryParam("filename") String fileName, // should be the name as
                                                     // how it lives in
                                                     // geoserver_data_dir,
                                                     // csv!!
            // @QueryParam("layername") String readableLayerName,
            @QueryParam("filetype") String fileType,
            @QueryParam("columntypes") String optColumnTypes // optional
                                                             // comma-separated
                                                             // string with
                                                             // postgres column
                                                             // types, e.g.
                                                             // real,text,text,integer...
    ) throws Exception {

        Integer layerId = null;
        try {
            layerId = Integer.valueOf(layerName.substring(layerName.lastIndexOf("_") + 1));
        } catch (Exception e) {
            throw new IllegalArgumentException("Invalid layer name " + layerName + " given. Must be in format "
                    + GeorepoDatasource.georepoDatasourcePrefix + "xxxx.");
        }

        if (fileType.equals("geocsv")) {
            return GeorepoTools.replaceGeoCSVLayer(
                    ctx,
                    ctx.getInitParameter(GeorepoConstants.geoserverDataDir),
                    sridAsString,
                    delimiter,
                    fileName,
                    workspace,
                    optColumnTypes,
                    layerId);
        } else {
            throw new IllegalArgumentException("Parameter filetype is not a supported filetype: " + fileType);
        }

    }

    @DELETE
    @Path("/{layername}")
    @Operation(summary = "Remove a layer from the workspace.")
    public static Response removeLayer(
            @PathParam("workspace") String workspace,
            @PathParam("layername") String layername) {

        // TODO

        return Response.status(Status.NOT_IMPLEMENTED).build();
    }

    public static void importFile0(ServletContext ctx, String workspace, String fileName,
            String layerName, String optColumnTypes) throws Exception {

        String epsg = "4326"; // currently expected to be wgs84
        String csvDelimiter = ";";

        String fileType = "";
        String fileExt = fileName.substring(fileName.lastIndexOf(".") + 1);
        System.out.println("importing: " + fileName);
        System.out.println("fileextension: " + fileExt);
        if (fileExt.equals("csv")) {
            fileType = "geocsv";
        }

        LayerController.createLayer(ctx, workspace, epsg, csvDelimiter, fileName, layerName, fileType, null,
                optColumnTypes);

    }
}
