package de.narimo.georepo.server.db;

import java.io.File;
import java.io.IOException;
import java.util.HashSet;
import java.util.Set;

import com.csvreader.CsvReader;

import de.narimo.commons.jdbc.JDBCConnectionJNDI;
import de.narimo.georepo.server.GeorepoConstants;
import de.narimo.georepo.server.GeorepoTools;
import de.narimo.georepo.server.tools.QueryCheck;

public class PostgisInput {

    /**
     * NOT in USE.
     *
     * Reads a shp file in two steps into postgis First creates a sql file from the
     * shape, then reads the sql file into a postgis table
     *
     * @see http://www.gistutor.com/postgresqlpostgis/4/18-importing-shapefile-gis-data-into-postgresql.html
     *
     *      use .pgpass file or some other method to allow database access.
     * @see http://stackoverflow.com/questions/9736085/run-a-postgresql-sql-file-using-command-line-args
     *
     * @author Ulrich Mann
     *
     */
    // public static void shp2postgis(File shpFile, String tableName, long
    // userId, int inSRID) throws Exception{
    //
    // String dbName = "geosky_data"; //pg data database name
    // String dbUser = "geosky_admin";
    // String dbPort = "5432";
    // String dbHost = "localhost";
    //
    //// String shpFileName = shpFile.getName();
    //
    //// String outTableName = "g_d_1"; //autoincrement index from meta table
    //// String fileName = shpFileName.substring(0,
    // shpFileName.lastIndexOf("."));
    //
    //// String inShpFile =
    // geoserverDataDir+"/data/"+userId+"/upload/"+shpFileName;
    //// String sqlFile =
    // geoserverDataDir+"/data/"+userId+"/sql/"+fileName+".sql";
    //
    // String sqlFile = shpFile.getAbsolutePath().substring(0,
    // shpFile.getAbsolutePath().lastIndexOf("."))+".sql";
    //
    // //create intermediate sql file
    //// String command = "shp2pgsql -s "+String.valueOf(inSRID)+" "+inShpFile+"
    // "+tableName+" "+dbName+" > "+sqlFile;
    // String command = "shp2pgsql -s "+String.valueOf(inSRID)+"
    // "+shpFile.getAbsolutePath()+" "+tableName+" "+dbName+" > "+sqlFile;
    //
    // System.out.println(command);
    //
    // Process p = Runtime.getRuntime().exec(command);
    // p.waitFor();
    //
    // //insert data table
    //// psql -d gisdatabase -U username -h hostname -p port -w -f parcels.sql
    // command = "psql -d "+dbName+" -U "+dbUser+" -h "+dbHost+" -p "+dbPort+"
    // -w -f "+sqlFile;
    //
    // System.out.println(command);
    //
    // p = Runtime.getRuntime().exec(command);
    // p.waitFor();
    //
    // }

    /**
     * Reads a geocsv file into a postgis table.
     *
     * @throws Exception
     * @see http://giswiki.hsr.ch/GeoCSV
     *
     *      Must contain a header line and at least following columns - WKT (for wkt
     *      geometries) - longitude, latitude (for point geometries) - lon, lat -
     *      long, lat - x, y
     */
    public static void csv2postgis(
            JDBCConnectionJNDI gsdataConn,
            String absCsvFileName,
            String tableName,
            char delimiter,
            String geoserverDataDir,
            int srid,
            String[] optColumnTypes) throws Exception {

        QueryCheck.checkTableName(tableName);

        CsvReader r = null;
        String tmpTableName = "temp_import_" + tableName;

        // final String GRPIDCOLUMN = "grpfid"; // georepo default id column for
        // // all tables. check that doesnt
        // // exist within user input columns
        //
        // final String gfidColumn = "gfid"; // georepo default name for real
        // world
        // // feature id

        // 1: create table

        try {

            r = new CsvReader(absCsvFileName, delimiter);
            r.readHeaders();

            // TODO: is that really expected for the real import?
//            gsdataConn.execute("DROP TABLE IF EXISTS " + tableName + "; ");
//            gsdataConn.execute("DROP TABLE IF EXISTS " + tmpTableName + "; ");

            StringBuilder sb = new StringBuilder();
            sb.append("CREATE TEMPORARY TABLE " + tmpTableName + " (");

            int i = 0;
            for (String header : r.getHeaders()) {
                header = normalizeHeader(header);

                // csvHeader.put(header, "text"); //TODO: insert everything just
                // as plain text? (preferred)

                if (i > 0) {
                    sb.append(", ");
                }
                sb.append(getHeaderStatementPart(header, optColumnTypes == null ? null : optColumnTypes[i]));
                i++;
            }

            sb.append(");");
            System.out.println(sb.toString());
            gsdataConn.execute(sb.toString());

        } finally {
            if (r != null) {
                r.close();
            }
        }

        // 2: fill table with file contents
        // NOTE: only allowed as super user
        // NOTE: currently uses semi-colon as hard-coded csv delimiter
        // String sql = "COPY "+tableName+" FROM '"+csvAbsFileName+"' DELIMITERS
        // '"+delimiter+"' CSV HEADER";
        String sql = "SELECT import_csv_file_to_temp_table('" + tmpTableName + "', '"
                + new File(absCsvFileName).getName() + "', '" + delimiter + "', '" + geoserverDataDir + "');";
        System.out.println(sql);
        gsdataConn.execute(sql);

        // make table permanent- temp table will be dropped automatically at
        // session end
        String tsql = "SELECT * INTO " + tableName + " FROM " + tmpTableName + ";";
        // String tsql = "CREATE TABLE "+tableName+" AS SELECT * FROM
        // "+tmpTableName+";";
        System.out.println(tsql);
        gsdataConn.execute(tsql);

        prepareGeographicTable(gsdataConn, tableName, srid, true);
    }

    /**
     * Creates a new data table with no rows.
     * 
     * @param gsdataConn
     * @param tableName
     * @param srid
     * @param headers
     * @param optColumnTypes
     * @throws Exception
     */
    public static void structure2postgis(
            JDBCConnectionJNDI gsdataConn,
            String tableName,
            int srid,
            String[] headers,
            String[] optColumnTypes) throws Exception {

        QueryCheck.checkTableName(tableName);

        StringBuilder sb = new StringBuilder();
        sb.append("CREATE TABLE " + tableName + " (");

        int i = 0;
        for (String header : headers) {
            header = normalizeHeader(header);

            if (i > 0) {
                sb.append(", ");
            }
            sb.append(getHeaderStatementPart(header, optColumnTypes == null ? null : optColumnTypes[i]));
            i++;
        }

        sb.append(");");
        System.out.println(sb.toString());
        gsdataConn.execute(sb.toString());

        prepareGeographicTable(gsdataConn, tableName, srid, false);
    }

    private static void prepareGeographicTable(JDBCConnectionJNDI gsdataConn, String tableName, int srid,
            boolean hasData)
            throws Exception {
        // ADD A SERIAL COLUMN
        String tsql = "ALTER TABLE " + tableName + " ADD COLUMN " + GeorepoConstants.GRPFIDCOLUMN
                + " serial PRIMARY KEY NOT NULL;";
        System.out.println(tsql);
        gsdataConn.execute(tsql);

        // ADD a gfid column (real world feature id)
        String gfidsql = "ALTER TABLE " + tableName + " ADD COLUMN " + GeorepoConstants.GFIDCOLUMN
                + " serial NOT NULL;";
        System.out.println(gfidsql);
        gsdataConn.execute(gfidsql);

        // attempt to fill empty gfid ids. We might run into a unique constraint
        // violation, in which case the customer data had missing id values that
        // now conflict with pre-entered gfids.
        // String gfidsqlupdate = "UPDATE " + tableName + " SET gfid=" +
        // GeorepoConstants.GRPFIDCOLUMN
        // + " WHERE " + GeorepoConstants.GFIDCOLUMN + " is null;";
        // System.out.println(gfidsqlupdate);
        // gsdataConn.execute(gfidsqlupdate);

        // 3: add geometry column
        // add the_geom column from either
        // x and y columns
        // lat lon columns
        // wkt column
        String schema = "public";
        // int srid = 4326;
        String geometryType = "POINT";
        int dimension = 2;

        String geomSql = "SELECT AddGeometryColumn ('" + schema + "','" + tableName + "','"
                + GeorepoConstants.GEOMETRYCOLUMN + "',"
                + String.valueOf(srid) + ",'" + geometryType + "'," + dimension + ");";
        System.out.println(geomSql);
        gsdataConn.execute(geomSql);

        if (hasData) {
            // 4: update the_geom column values
            String lon = "lon"; // name of longitude column that is used in table
            String lat = "lat"; // name of latitude column that is used in table
            String updateSql = "";
            if (geometryType.equals("POINT")) {
                // updateSql = "UPDATE "+tableName+" SET the_geom =
                // ST_GeomFromText('POINT("+lon+" "+lat+")');"; //TODO: update
                // column, fill with geometries
                updateSql = "UPDATE " + tableName + " SET " + GeorepoConstants.GEOMETRYCOLUMN
                        + " = ST_SetSRID(ST_MakePoint(cast(" + lon
                        + " as float), cast(" + lat + " as float)), " + srid + ");";

            } else if (geometryType.equals("POLYGON")) {
                // TODO thats just wkt option!
            } else if (geometryType.equals("LINESTRING")) {
                // TODO thats just wkt option!
            }
            System.out.println(updateSql);
            gsdataConn.execute(updateSql);
        }
    }

    private static String getHeaderStatementPart(String header, String columnType) {
        if (columnType == null) {
            return "\"" + header + "\" " + "text"; // so far insert only as text
        } else {
            // allow certain types for columns
            return "\"" + header + "\" " + (isAllowedColumnType(columnType) ? columnType : "text");
        }
    }

    private static String normalizeHeader(String header) throws IOException {
        if (header.equals(GeorepoConstants.GEOMETRYCOLUMN) || // reserved keyword for geometry column
                header.equals(GeorepoConstants.GRPFIDCOLUMN) || // reserved keyword for georepo grpfid column
                header.equals(GeorepoConstants.GFIDCOLUMN)) { // reserved keyword for georepo gfid column
            header = header + "_original";
        }

        if (!GeorepoTools.isAlphaNumeric(header)) {
            header = GeorepoTools.normalize(header);
            System.out.println(
                    "Header field must only contain alphanumeric and underscore characters. Normalized to "
                            + header);
            // throw new IOException("Header fields must only contain
            // alphanumeric and underscore characters.");
        }

        // grs-188: lower case column names
        return header.toLowerCase();
    }

    /**
     * Do spatial optimizations like spatial indexing, analyze, clustering on new
     * geometry tables to enhance query speed.
     *
     * @throws Exception
     */
    public static void optimize(JDBCConnectionJNDI gsdataConn, String tableName, String geometryColumnName)
            throws Exception {

        if (geometryColumnName == null) {
            geometryColumnName = GeorepoConstants.GEOMETRYCOLUMN;
        }

        // drop spatial index on geometry table
        String sql = "DROP INDEX IF EXISTS " + tableName + "_idx;";
        System.out.println(sql);
        gsdataConn.execute(sql);

        // (re)create spatial index on geometry table
        sql = "CREATE INDEX " + tableName + "_idx ON " + tableName + " USING GIST (" + geometryColumnName + ");";
        System.out.println(sql);
        gsdataConn.execute(sql);

        // vacuum to clear unused table space
        // analyze to calc internal statistics
        sql = "ANALYZE " + tableName + ";";
        System.out.println(sql);
        gsdataConn.execute(sql);

        // cluster table to sort entries based on geographic location/ distance
        sql = "CLUSTER " + tableName + " USING " + tableName + "_idx;";
        System.out.println(sql);
        gsdataConn.execute(sql);

    }

    public static void vacuum(JDBCConnectionJNDI gsdataConn, String tableName) throws Exception {

        QueryCheck.checkTableName(tableName);

        // vacuum to clear unused table space
        // analyze to calc internal statistics
        String sql = "VACUUM " + tableName + ";";
        System.out.println(sql);
        gsdataConn.execute(sql);

    }

    private static boolean isAllowedColumnType(String columnType) {

        if (columnType == null) {
            return false;
        }

        Set<String> allowedColumnTypes = new HashSet<>();
        allowedColumnTypes.add("text");
        allowedColumnTypes.add("real");
        allowedColumnTypes.add("double precision");
        allowedColumnTypes.add("smallint");
        allowedColumnTypes.add("integer");
        allowedColumnTypes.add("bigint");
        allowedColumnTypes.add("boolean");
        allowedColumnTypes.add("timestamp");

        if (allowedColumnTypes.contains(columnType)) {
            return true;
        }
        return false;
    }
}
