package de.narimo.georepo.server.filter;

import java.io.IOException;
import java.util.List;

import javax.annotation.Priority;
import javax.servlet.FilterConfig;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.HttpMethod;
import javax.ws.rs.Priorities;
import javax.ws.rs.container.ContainerRequestContext;
import javax.ws.rs.container.ContainerRequestFilter;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.PathSegment;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.UriInfo;
import javax.ws.rs.ext.Provider;

import de.narimo.geocore.ws.repository.UserRepository;
import de.narimo.georepo.server.repository.WorkspaceRepository;

/**
 * Authorization filter allowing or disallowing access to a secured workspace
 * for authenticated users.
 * Needs to be executed after Authentication by BasicAuthenticationFilter and
 * thus is dependent on it.
 *
 * @author u.mann
 *
 */
@Provider
@Priority(Priorities.AUTHORIZATION)
public class WorkspaceAuthorizationFilter implements ContainerRequestFilter {

    @Context
    private HttpServletRequest httpServletRequest;

    @Context
    private ServletContext ctx;

    @Context
    private UriInfo uriinfo;

    @Override
    public void filter(ContainerRequestContext crc) throws IOException {

        System.out.println(this.getClass().getSimpleName());
        System.out.println("Request method: " + this.httpServletRequest.getMethod());

        try {

            if (this.httpServletRequest.getMethod().equals("OPTIONS")) {
                return;
            }

            List<PathSegment> pathSegments = this.uriinfo.getPathSegments();

            if (pathSegments.size() > 0 && pathSegments.get(0) != null && pathSegments.get(0).toString().equals("workspaces")) {
                String uniqueUsername = crc.getSecurityContext().getUserPrincipal().getName();
                if (uniqueUsername == null) {
                    throw new SecurityException("Unknown user principal " + uniqueUsername);
                }

                // beware of changing api path's, this will fail!
                String requestedWorkspace = (pathSegments.size() > 1 && pathSegments.get(1) != null)
                        ? pathSegments.get(1).toString() : null;
                if (requestedWorkspace == null) {
                    // allow without authorization
                    return;
                }

                // registration endpoint is defined in the WorkspaceController
                boolean workspaceRegistrationRequest = false;
                // try {
                    workspaceRegistrationRequest = pathSegments.size() > 2 && pathSegments.get(2) != null
                        && pathSegments.get(2).toString().equals("register");
                    System.out.println("register request?: " + workspaceRegistrationRequest);
                // } catch (IndexOutOfBoundsException e) {
                    // no path segment after workspace, which is valid
                    // use normal authorization in that case
                // }
                if (workspaceRegistrationRequest) {
                    // allow without authorization
                    return;
                }

                // TODO: is that check ok?? or rather risky?
                boolean write = this.httpServletRequest.getMethod().equals(HttpMethod.GET) ? false : true;

                System.out.println("user " + uniqueUsername + " allowed to " + (write ? "write" : "read")
                        + " requested workspace " + requestedWorkspace + "?");
                int workspaces = getAllowedWorkspaceCount(requestedWorkspace, uniqueUsername, write);
                if (workspaces <= 0) {
                    System.out.println("no");
                    reject(crc); // reject if no workspaces exist with requested
                                 // rights
                }
                System.out.println("yes");
                return;
            }
        } catch (Exception e) {
            e.printStackTrace();
            reject(crc);
        }
    }

    /**
     * Get the workspace allowed for a username.
     *
     * @param jdbcMeta
     * @param workspace
     * @param username
     * @param write
     * @return
     * @throws Exception
     */
    private static int getAllowedWorkspaceCount(String workspace, String username, boolean write) {

        String permission = write == true ? "rw" : "r";

        Integer userId = null;
        try {
            userId = UserRepository.getUserId(username);
        } catch (Exception e) {
            System.out.println("Cannot get allowed workspace for user " + username + ";");
            return 0;
        }

        if (userId == null) {
            System.out.println("Cannot get allowed workspace for user " + username + ";");
            return 0;
        }

        return WorkspaceRepository.getWorkspaceCount(userId, workspace, permission);
    }

    /**
     * Reject an authorization attempt to disallowed workspace.
     *
     * @param crc
     * @param includeBasicAuthHeader
     */
    private static void reject(ContainerRequestContext crc) {
        try {
            System.out.println("Authorization to workspace failed.");
            crc.abortWith(Response
                    .status(Response.Status.FORBIDDEN)
                    .entity("Authorization failed. Access denied.")
                    .type(MediaType.TEXT_PLAIN)
                    .build());

        } catch (Exception e) {
            return;
        }
        return;
    }

    public void init(FilterConfig filterConfig) throws ServletException {
    }

    public void destroy() {
    }
}
