package de.narimo.georepo.server.geoserver;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.simpleframework.xml.Serializer;
import org.simpleframework.xml.core.Persister;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonMappingException;

import de.narimo.commons.http.URLResponse;
import de.narimo.commons.json.JsonConverter;
import de.narimo.commons.ws.http.HttpMethod;
import de.narimo.commons.ws.http.HttpURLClient;
import de.narimo.georepo.server.GeorepoConstants;
import de.narimo.georepo.server.dto.DataStore;
import de.narimo.georepo.server.dto.DataStoreXML;
import de.narimo.georepo.server.dto.DataStores;
import de.narimo.georepo.server.dto.DataStoresList;
import de.narimo.georepo.server.dto.FeatureType;
import de.narimo.georepo.server.dto.FeatureTypes;
import de.narimo.georepo.server.dto.FeatureTypesList;

public class GeoserverWorkspace {

    private String geoserverRestUrl;
    private Map<String, String> headers;
    private String geoserverAdmin;
    private String geoserverPass;

    private String workspace;

    public GeoserverWorkspace(String geoserverRestUrl, Map<String, String> headers, String geoserverAdmin,
            String geoserverPass) {

        this.geoserverRestUrl = geoserverRestUrl;
        this.headers = headers;
        this.geoserverAdmin = geoserverAdmin;
        this.geoserverPass = geoserverPass;
    }

    public GeoserverWorkspace(String workspaceName, String geoserverRestUrl, Map<String, String> headers,
            String geoserverAdmin, String geoserverPass) {

        this.workspace = workspaceName;
        this.geoserverRestUrl = geoserverRestUrl;
        this.headers = headers;
        this.geoserverAdmin = geoserverAdmin;
        this.geoserverPass = geoserverPass;
    }

    /**
     * Creates a workspace by creating a namespace which auto-creates a bare
     * workspace.
     *
     * @param workspaceName
     * @throws Exception
     */
    public void createWorkspace() throws IOException { this.createNamespace(this.workspace); }

    /**
     * Create a new namespace. This will automatically create a workspace too.
     *
     * @param namespace
     * @param nsUri
     * @throws Exception
     */
    public void createNamespace(String namespace) throws IOException {

        if (namespace == null) {
            throw new IllegalArgumentException("Cannot create namespace null.");
        }

        String namespaceUri = this.geoserverRestUrl + "/namespaces/" + namespace;

        String body = "<namespace><prefix>" + namespace + "</prefix><uri>" + namespaceUri + "</uri></namespace>";

        URLResponse rr = null;
        try {
            rr = HttpURLClient.sendRequest(this.geoserverRestUrl + "/namespaces", null, this.headers, body,
                    HttpMethod.POST, null, this.geoserverAdmin, this.geoserverPass, false, false);

        } catch (Exception e) {
            e.printStackTrace();
            throw new IOException(e);
        }
        if (rr.getStatusCode() < 200 || rr.getStatusCode() > 201) {
            throw new InternalError(
                    "Creation of namespace " + namespace + " failed with status " + rr.getStatusCode() + ".");
        }

    }

    public void modifyWorkspaceSettings(String charset, String proxyBaseUrl, Contact contact, String onlineResource,
            boolean enabled)
            throws IOException {

        WorkspaceSettings settings = new WorkspaceSettings();
        settings.setCharset(charset);
        settings.setContact(contact);
        settings.setProxyBaseUrl(proxyBaseUrl);
        settings.setOnlineResource(onlineResource);

        String body = JsonConverter.toJson(settings);

        System.out.println("Sending settings json to workspace:\n" + body);
        String url = this.geoserverRestUrl + "/workspaces/" + workspace + "/settings";

        Map<String, String> headers = new HashMap<String, String>();
        headers.put("Content-Type", "application/json");

        URLResponse rr = HttpURLClient.sendRequest(url, null, headers, body, HttpMethod.PUT, null,
                this.geoserverAdmin, this.geoserverPass, false, false);

        if (rr.getStatusCode() < 200 || rr.getStatusCode() > 201) {
            throw new InternalError(
                    "Modification of workspace " + workspace + " failed with status " + rr.getStatusCode() + ".");
        }
    }

    public void modifyWorkspaceOWSSettings(String service, ServiceSettings serviceSettings)
            throws IOException {

        String body = JsonConverter.toJson(serviceSettings);

        System.out.println("Sending settings json to service:\n" + body);
        String url = this.geoserverRestUrl + "/services/" + service + "/workspaces/" + workspace + "/settings";

        Map<String, String> headers = new HashMap<String, String>();
        headers.put("Content-Type", "application/json");

        URLResponse rr = HttpURLClient.sendRequest(url, null, headers, body, HttpMethod.PUT, null,
                this.geoserverAdmin, this.geoserverPass, false, false);

        if (rr.getStatusCode() < 200 || rr.getStatusCode() > 201) {
            throw new InternalError(
                    "Modification of workspace " + workspace + " failed with status " + rr.getStatusCode() + ".");
        }
    }

    public String getWorkspaceOWSSettings(String service) throws Exception {

        String url = this.geoserverRestUrl + "/services/" + service + "/workspaces/" + workspace + "/settings";

        Map<String, String> headers = new HashMap<String, String>();
        headers.put("Content-Type", "application/json");

        URLResponse rr = HttpURLClient.sendRequest(url, null, headers, null, HttpMethod.GET, null,
                this.geoserverAdmin, this.geoserverPass, false, false);

        if (rr.getStatusCode() < 200 || rr.getStatusCode() > 201) {
            throw new InternalError("Retrieval of workspace " + workspace + " settings failed with status "
                    + rr.getStatusCode() + ".");
        }

        return rr.getResponseBody();

    }

    /**
     * Change transaction operation enabled in wfs settings. Can be BASIC := read
     * access, TRANSACTIONAL := write access, COMPLETE := read+write access
     *
     * Workspace must be enabled to do this!
     *
     * @param transactional
     */
    public void setTransactional(boolean transactional, String workspace) throws IOException {

        String body = null;

        if (transactional) {
            body = "<wfs><serviceLevel>COMPLETE</serviceLevel></wfs>";
        } else {
            body = "<wfs><serviceLevel>BASIC</serviceLevel></wfs>";
        }

        // if(transactional){
        // body = "<workspace><name>COMPLETE</name></workspace>";
        // }else{
        // body = "<wfs><serviceLevel>BASIC</serviceLevel></wfs>";
        // }

        URLResponse rr = null;
        try {
            rr = HttpURLClient.sendRequest(
                    this.geoserverRestUrl + "/services/wfs/workspaces/" + workspace + "/settings", null, this.headers,
                    body, HttpMethod.POST, null, this.geoserverAdmin, this.geoserverPass, false, false);

        } catch (Exception e) {
            e.printStackTrace();
            throw new IOException(e);
        }
        if (rr.getStatusCode() < 200 || rr.getStatusCode() > 201) {
            throw new IOException("Enabling/ Disabling of transaction support for workspace " + workspace
                    + " failed with status " + rr.getStatusCode() + ".");
        }
    }

    /**
     * Returns all datasource names, that exist on this workspace
     *
     * @param geoserverRestUrl
     * @param workspace
     * @return
     * @throws Exception
     */
    public List<String> getDatasources(String workspace) throws IOException {

        String url = "/workspaces/" + workspace + "/datastores.json";

        URLResponse rr = HttpURLClient.sendRequest(this.geoserverRestUrl + url, null, null, null, HttpMethod.GET, null,
                this.geoserverAdmin, this.geoserverPass, false, false);

        ArrayList<String> datasources = new ArrayList<String>();

        try {
            DataStoresList sl = JsonConverter.fromJson(rr.getResponseBody(), DataStoresList.class);

            // ArrayList<String> datasources = new ArrayList<String>();

            for (DataStores f : sl.getDataStores()) {
                for (DataStore ds : f.getDataStore()) {
                    System.out.println(ds.getName());
                    datasources.add(ds.getName());
                }
            }

        } catch (JsonMappingException jme) {
            // no datasources exist or malformed?
            // return null;
        }

        return datasources;
    }

    /**
     * Gets the first available datasource on this workspace, that is of type
     * datasourceType.
     *
     * @param workspace
     * @param datasourceType
     * @return
     * @throws Exception
     */
    public String getDatasource(String workspace, GeoserverDatasourceType datasourceType) throws IOException {

        Serializer serializer = new Persister();
        List<String> datasourceNames = this.getDatasources(workspace);

        for (String ds : datasourceNames) {

            String datasourceXML = this.getDatasourceXML(workspace, ds);
            // use strict=false to ignore unused elements
            // according to
            // https://stackoverflow.com/questions/4740934/how-to-ignore-unused-xml-elements-while-deserializing-a-document
            DataStoreXML dataStore;
            try {
                dataStore = serializer.read(DataStoreXML.class, datasourceXML, false);
            } catch (Exception e) {
                e.printStackTrace();
                throw new IOException(e);
            }

            String datasourceTypePgJNDI = datasourceType.toString();
            if (datasourceType == GeoserverDatasourceType.PostGISJNDI) {
                datasourceTypePgJNDI = "PostGIS (JNDI)";
            }

            if (dataStore.getType().equals(datasourceTypePgJNDI)
            // && dataStore.isEnabled() //TODO: should we use enabled
            // datasources only??
            ) {
                return dataStore.getName();
            }
        }

        return null;
    }

    /**
     * Get vector datasources that exist in the given workspace.
     *
     * @param workspace
     * @param datasource
     * @throws Exception
     */
    public String getDatasourceXML(String workspace, String datasource) throws IOException {

        if (datasource == null) {
            throw new IOException("Missing value datasource.");
        }

        URLResponse rr = HttpURLClient.sendRequest(
                this.geoserverRestUrl + "/workspaces/" + workspace + "/datastores/" + datasource + ".xml", null, null,
                null, HttpMethod.GET, null, this.geoserverAdmin, this.geoserverPass, false, false);

        String datasourcesXML = rr.getResponseBody();

        return datasourcesXML;
    }

    /**
     * Retrieves all feature types/ layers per workspace.
     * 
     * @param workspace
     * @return list of feature types on this workspace
     * @throws IOException
     */
    public List<FeatureType> getWorkspaceFeatureTypes(String workspace, boolean includeDiffLayers) throws IOException {

        if (workspace == null) {
            throw new IOException("Missing value workspace.");
        }

        URLResponse rr = HttpURLClient.sendRequest(
                this.geoserverRestUrl + "/workspaces/" + workspace + "/featuretypes.json", null, null, null,
                HttpMethod.GET, null, this.geoserverAdmin, this.geoserverPass, false, false);

        List<FeatureType> featureTypes = new ArrayList<FeatureType>();

        try {
            FeatureTypesList ftl = JsonConverter.fromJson(rr.getResponseBody(), FeatureTypesList.class);

            for (FeatureTypes fts : ftl.featureTypes) {
                for (FeatureType ft : fts.featureType) {
                    if (!includeDiffLayers && ft.getName().endsWith(GeorepoConstants.diffTableSuffix)) {
                        continue;
                    }
                    ft.setHref(null);
                    ft.setDescription(null);
                    featureTypes.add(ft);
                }
            }
            featureTypes.sort((o1, o2) -> o1.getName().compareTo(o2.getName()));
        } catch (JsonMappingException jme) {
            // no layers exist or malformed?
        }
        return featureTypes;
    }

    /**
     * Creates a WFS service which is disabled by default.
     * 
     * @param workspace
     * @throws IOException
     */
    public void createWFSService() throws IOException {

        if (workspace == null) {
            throw new IllegalArgumentException("Cannot create WFS service for workspace null.");
        }

        String url = this.geoserverRestUrl + "/services/wfs/workspaces/" + workspace + "/settings.xml";
        File resource = new File(this.getClass().getResource("wfs-raw.xml").getFile());
        String body = new String(Files.readAllBytes(resource.toPath()));

        URLResponse rr = null;
        try {
            rr = HttpURLClient.sendRequest(url, null, this.headers, body,
                    HttpMethod.PUT, null, this.geoserverAdmin, this.geoserverPass, false, false);
        } catch (Exception e) {
            e.printStackTrace();
            throw new IOException(e);
        }
        if (rr.getStatusCode() < 200 || rr.getStatusCode() > 201) {
            throw new InternalError(
                    "Creation of wfs service for workspace " + workspace + " failed with status " + rr.getStatusCode()
                            + ".");
        }
    }

    public void createWMSService() throws IOException {

        if (workspace == null) {
            throw new IllegalArgumentException("Cannot create WMS service for workspace null.");
        }

        String url = this.geoserverRestUrl + "/services/wms/workspaces/" + workspace + "/settings.xml";
        File resource = new File(this.getClass().getResource("wms-raw.xml").getFile());
        String body = new String(Files.readAllBytes(resource.toPath()));

        URLResponse rr = null;
        try {
            rr = HttpURLClient.sendRequest(url, null, this.headers, body,
                    HttpMethod.PUT, null, this.geoserverAdmin, this.geoserverPass, false, false);
        } catch (Exception e) {
            e.printStackTrace();
            throw new IOException(e);
        }
        if (rr.getStatusCode() < 200 || rr.getStatusCode() > 201) {
            throw new InternalError(
                    "Creation of wms service for workspace " + workspace + " failed with status " + rr.getStatusCode()
                            + ".");
        }
    }

    public static void main(String[] args) throws JsonProcessingException {
        ServiceSettings settings = new ServiceSettings();
        settings.setTitle("title");
        settings.setEnabled(true);
        settings.setAbstrct("");
        settings.setAccessConstraints("");
        settings.setMaintainer("");

        Keywords k = new Keywords();
        k.setString(null);
        settings.setKeywords(k);

        System.out.println(JsonConverter.toJson(settings));
    }
}
