package de.narimo.georepo.server.repository;

import java.io.IOException;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import de.narimo.commons.jdbc.JDBCConnectionJNDI;
import de.narimo.georepo.server.tools.WorkspacePermissions;

public class WorkspaceRepository {

    public static void changeWorkspacePermission(int userid, String workspace, String permission)
            throws SQLException, IOException {

        JDBCConnectionJNDI jdbcMeta = null;

        try {
            jdbcMeta = new JDBCConnectionJNDI("jdbc/georepoMetaResource");
            jdbcMeta.setAutoCommit(false);

            String delSQL = "DELETE FROM workspaces WHERE userid=? AND workspace=?;";
            System.out.println(delSQL);

            PreparedStatement ps0 = jdbcMeta.prepareStatement(delSQL);
            ps0.setInt(1, userid);
            ps0.setString(2, workspace);
            ps0.executeUpdate();
            System.out.println(ps0.toString());
            if (!permission.equals(WorkspacePermissions.noPermission)) {

                // Set the new permission, if any
                String sql = "INSERT INTO workspaces (userid, workspace, rights) VALUES (?,?,?)";
                System.out.println(sql);
                PreparedStatement ps1 = jdbcMeta.prepareStatement(sql);
                ps1.setInt(1, userid);
                ps1.setString(2, workspace);
                ps1.setString(3, permission);
                ps1.execute();
                System.out.println(ps1.toString());
            }

            jdbcMeta.commit();

        } catch (Exception e) {
            e.printStackTrace();
            jdbcMeta.rollback();
            throw new IOException(
                    "Could not set workspace permissions for workspace " + workspace + " and user " + userid + ".");
        } finally {
            if (jdbcMeta != null) {
                jdbcMeta.closeAll();
                jdbcMeta = null;
            }
        }
    }

    public static int getWorkspaceCount(int userId, String workspace, String permission) {
        JDBCConnectionJNDI jdbcMeta = null;

        try {
            jdbcMeta = new JDBCConnectionJNDI("jdbc/georepoMetaResource");

            String sql1 = "SELECT count(*) FROM workspaces w WHERE " + "w.userid = ? AND " + "w.workspace = ? AND "
                    + "w.rights LIKE ?;";
            System.out.println(sql1);

            PreparedStatement ps = jdbcMeta.prepareStatement(sql1);
            ps.setInt(1, userId);
            ps.setString(2, workspace);
            ps.setString(3, "%" + permission + "%");

            ResultSet rs1 = ps.executeQuery();
            rs1.next();
            return rs1.getInt("count");

        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (jdbcMeta != null) {
                jdbcMeta.closeAll();
                jdbcMeta = null;
            }
        }
        return 0;
    }

    public static List<Integer> getWorkspaceAdminIds(String workspace) {
        JDBCConnectionJNDI jdbcMeta = null;
        List<Integer> adminIds = new ArrayList<Integer>();
        try {
            jdbcMeta = new JDBCConnectionJNDI("jdbc/georepoMetaResource");

            String permission = WorkspacePermissions.adminPermission;
            String sql1 = "SELECT userid FROM workspaces w WHERE " + "w.workspace = ? AND " + "w.rights LIKE ?;";

            PreparedStatement ps = jdbcMeta.prepareStatement(sql1);
            ps.setString(1, workspace);
            ps.setString(2, "%" + permission + "%");

            ResultSet rs1 = ps.executeQuery();
            while (rs1.next()) {
                adminIds.add(rs1.getInt("userid"));
            }
        } catch (Exception e) {
            e.printStackTrace();
            throw new InternalError();
        } finally {
            if (jdbcMeta != null) {
                jdbcMeta.closeAll();
                jdbcMeta = null;
            }
        }
        return adminIds;
    }

    /**
     * Returns the type of permission for a specific workspace.
     *
     * @param userId
     * @param workspace
     * @return
     */
    public static String getWorkspacePermissionType(int userId, String workspace) {
        JDBCConnectionJNDI jdbcMeta = null;
        try {
            jdbcMeta = new JDBCConnectionJNDI("jdbc/georepoMetaResource");

            String sql1 = "SELECT rights FROM workspaces w WHERE " + "w.workspace = ? AND " + "w.userid = ? LIMIT 1;";

            PreparedStatement ps = jdbcMeta.prepareStatement(sql1);
            ps.setString(1, workspace);
            ps.setInt(2, userId);

            ResultSet rs = ps.executeQuery();

            String rights = WorkspacePermissions.noPermission;

            int count = 0;
            while (rs.next()) {
                count++;
                rights = rs.getString("rights");
            }
            if (count > 1) {
                throw new IllegalArgumentException("Inconsistent entries for workspace permissions. " + count
                        + " different permissions exist.");
            }
            return rights;
        } catch (Exception e) {
            e.printStackTrace();
            throw new InternalError();
        } finally {
            if (jdbcMeta != null) {
                jdbcMeta.closeAll();
                jdbcMeta = null;
            }
        }
    }

    /**
     * Confirm that a given secrect conforms with the server side workspace secret.
     * Used e.g. to authorize workspace registration.
     *
     * @param workspace
     * @param workspaceSecret
     * @return
     */
    public static boolean isWorkspaceSecretValid(String workspace, String workspaceSecret) {
        JDBCConnectionJNDI jdbcMeta = null;
        try {
            jdbcMeta = new JDBCConnectionJNDI("jdbc/georepoMetaResource");

            String sql1 = "SELECT * FROM workspacesecrets w WHERE " + "w.workspace = ? AND w.secret = ?;";

            PreparedStatement ps0 = jdbcMeta.prepareStatement(sql1);
            ps0.setString(1, workspace);
            ps0.setString(2, workspaceSecret);
            ResultSet rs = ps0.executeQuery();

            if (rs.next()) {
                return true;
            }
            return false;
        } catch (Exception e) {
            e.printStackTrace();
            throw new InternalError();
        } finally {
            if (jdbcMeta != null) {
                jdbcMeta.closeAll();
                jdbcMeta = null;
            }
        }
    }

    public static Map<String, String> getWorkspaces(int userId, boolean includeDiffWorkspaces) {
        JDBCConnectionJNDI jdbcMeta = null;

        LinkedHashMap<String, String> workspaces = new LinkedHashMap<String, String>();

        try {
            jdbcMeta = new JDBCConnectionJNDI("jdbc/georepoMetaResource");

            // Better would be to have explicit workspaces and diffworkspace flags in a
            // separate table
            String sql = "SELECT distinct(workspace), rights FROM workspaces WHERE userid= ? "
                    + "AND workspace NOT IN(SELECT diffworkspace FROM difftables) ORDER BY workspace ASC;";

            if (includeDiffWorkspaces) {
                sql = "SELECT workspace, rights FROM workspaces WHERE userid = ? ORDER BY workspace ASC;";
            }

            PreparedStatement ps = jdbcMeta.prepareStatement(sql);
            ps.setInt(1, userId);

            ResultSet rs = ps.executeQuery();
            while (rs.next()) {
                String permission = WorkspacePermissions.getReadablePermission(rs.getString("rights"));
                workspaces.put(rs.getString("workspace"), permission);
            }
        } catch (Exception e) {
            e.printStackTrace();
            throw new InternalError();
        } finally {
            if (jdbcMeta != null) {
                jdbcMeta.closeAll();
                jdbcMeta = null;
            }
        }
        return workspaces;
    }
}
