package de.narimo.georepo.server.api;

import java.io.IOException;
import java.sql.SQLException;
import java.util.List;

import javax.servlet.ServletContext;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.SecurityContext;
import javax.ws.rs.ext.Provider;

import de.narimo.commons.dto.User;
import de.narimo.geocore.ws.repository.UserRepository;
import de.narimo.georepo.server.api.documents.DocumentType;
import de.narimo.georepo.server.api.documents.FeatureDocument;
import de.narimo.georepo.server.api.documents.FeatureDocumentResponse;
import de.narimo.georepo.server.api.documents.ImageAndDocumentService;
import de.narimo.georepo.server.repository.DatasetRepository;
import de.narimo.georepo.server.repository.DifftableRepository;
import de.narimo.georepo.server.tools.AdminTools;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;

@Provider
@Path("/workspaces/{diffWorkspace}/layers/{layername}/documents/changes")
@Tag(name = "Documents")
public class DocumentLayerModificationController {

    ImageAndDocumentService imageAndDocumentService = new ImageAndDocumentService();

    @GET
    @Path("/")
    @Produces(MediaType.APPLICATION_JSON)
    @Operation(summary = "Retrieves all unreviewed documents of a layer")
    public Response getLayerImages(
            @Context SecurityContext sec,
            @Context ServletContext ctx,
            @PathParam("diffWorkspace") String diffWorkspace,
            @PathParam("layername") String dataLayerName) {

        FeatureDocumentResponse response = new FeatureDocumentResponse();

        try {
            User user = UserRepository.getUser(sec.getUserPrincipal().getName()).get();

            String workspace = DifftableRepository.getDataWorkspace(diffWorkspace, dataLayerName);
            if (workspace == null) {
                throw new InternalError("No data workspace defined.");
            }

            // Does only admins should retrieve non-reviewed images?
            // As a user, I would like to retrieve my non-reviewed images
            AdminTools.checkAdminPermission(user.getId(), workspace);

            int layerId = DatasetRepository.getLayerId(workspace, dataLayerName);
            List<FeatureDocument> images = imageAndDocumentService.getDiffFeatureImagesOrDocuments(user,
                    workspace, layerId, DocumentType.DOCUMENT);

            response.setDocuments(images);

        } catch (SQLException | InternalError e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        } catch (IOException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).entity("Could not retrieve document names.").build();
        }
        return Response.ok(response).build();
    }
}
