package de.narimo.georepo.server.api;

import java.io.IOException;
import java.sql.SQLException;

import javax.servlet.ServletContext;
import javax.ws.rs.ForbiddenException;
import javax.ws.rs.GET;
import javax.ws.rs.NotFoundException;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.SecurityContext;
import javax.ws.rs.ext.Provider;

import de.narimo.commons.dto.User;
import de.narimo.geocore.ws.repository.UserRepository;
import de.narimo.georepo.server.layer.FeatureLayer;
import de.narimo.georepo.server.repository.DifftableRepository;
import de.narimo.georepo.server.tools.AdminTools;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;

@Provider
@Path("/workspaces/{workspace}/layers/{layername}/features")
@Tag(name = "Features")
public class FeatureController {

    @GET
    @Path("/")
    @Produces(MediaType.APPLICATION_JSON)
    @Operation(summary = "Retrieves all features from a layer")
    public static Response getFeatures(
            // should be determined from layerid
            @PathParam("workspace") String dataWorkspace,
            // should be determined from layerid
            @PathParam("layername") String dataLayerName, @Context SecurityContext sec, @Context ServletContext ctx) {

        try {

            // Features may not be retrieved from a diff table using this
            // endpoint
            // That is, double check if the requested workspace is a diff
            // workspace, then disallow!
            boolean isDiffWorkspace = isDiffWorkspace(dataWorkspace, dataLayerName);
            if (isDiffWorkspace) {
                throw new ForbiddenException("The given workspace is a diff workspace.");
            }

            User user = UserRepository.getUser(sec.getUserPrincipal().getName()).get();
            String jsonFeatureCollection = FeatureLayer.getFeatures(user.getId(), dataWorkspace, dataLayerName);

            if (jsonFeatureCollection == null) {
                throw new InternalError("Could not get feature changes.");
            }

            return Response.ok().entity(jsonFeatureCollection).build();
        } catch (NotFoundException e) {
            e.printStackTrace();
            return Response.status(Status.NOT_FOUND).build();
        } catch (IOException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        } catch (ForbiddenException e) {
            e.printStackTrace();
            return Response.status(Status.FORBIDDEN).build();
        } catch (SQLException | InternalError | RuntimeException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }

    @GET
    @Path("/{gfid}/history")
    @Produces(MediaType.APPLICATION_JSON)
    @Operation(summary = "Retrieves the history of a feature from a layer")
    public static Response getFeatureHistory(
            // should be determined from layerid
            @PathParam("workspace") String dataWorkspace,
            // should be determined from layerid
            @PathParam("layername") String dataLayerName,
            @PathParam("gfid") int gfid,
            @Context SecurityContext sec,
            @Context ServletContext ctx) {

        try {

            boolean isDiffWorkspace = isDiffWorkspace(dataWorkspace, dataLayerName);
            if (isDiffWorkspace) {
                throw new ForbiddenException("The given workspace is a diff workspace.");
            }

            User user = UserRepository.getUser(sec.getUserPrincipal().getName()).get();
            AdminTools.checkAdminPermission(user.getId(), dataWorkspace);

            String jsonFeatureCollection = FeatureLayer.getFeatureHistory(user.getId(), dataWorkspace, dataLayerName,
                    gfid);

            if (jsonFeatureCollection == null) {
                throw new InternalError("Could not get feature history.");
            }

            return Response.ok().entity(jsonFeatureCollection).build();
        } catch (NotFoundException e) {
            e.printStackTrace();
            return Response.status(Status.NOT_FOUND).build();
        } catch (IOException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        } catch (ForbiddenException e) {
            e.printStackTrace();
            return Response.status(Status.FORBIDDEN).build();
        } catch (SQLException | InternalError | RuntimeException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }

    /**
     * Is a workspace actually a diff workspace?
     * 
     * @param dataWorkspace
     * @param dataLayerName
     * @return
     */
    private static boolean isDiffWorkspace(String dataWorkspace, String dataLayerName) {
        String diffWorkspace = DifftableRepository.getDiffWorkspace(dataWorkspace);
        String diffLayerName = DifftableRepository.getDiffLayername(dataWorkspace, diffWorkspace, dataLayerName);
        return dataWorkspace.equals(diffWorkspace) || dataLayerName.equals(diffLayerName);
    }
}
