package de.narimo.georepo.server.api;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.nio.file.NoSuchFileException;
import java.sql.SQLException;
import java.util.List;

import javax.servlet.ServletContext;
import javax.ws.rs.GET;
import javax.ws.rs.NotFoundException;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.SecurityContext;
import javax.ws.rs.ext.Provider;

import de.narimo.commons.dto.User;
import de.narimo.geocore.ws.repository.UserRepository;
import de.narimo.georepo.server.api.documents.DocumentType;
import de.narimo.georepo.server.api.documents.FeatureDocument;
import de.narimo.georepo.server.api.documents.FeatureImageResponseLegacy;
import de.narimo.georepo.server.api.documents.ImageAndDocumentService;
import de.narimo.georepo.server.repository.DatasetRepository;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;

@Provider
@Path("/workspaces/{workspace}/layers/{layername}/features/{gfid}/images")
@Tag(name = "Images")
public class ImageController {

    ImageAndDocumentService imageAndDocumentService = new ImageAndDocumentService();

    @GET
    @Path("/")
    @Produces(MediaType.APPLICATION_JSON)
    @Operation(summary = "Retrieves all images of a feature")
    public Response getFeatureImages(
            @Context SecurityContext sec,
            @Context ServletContext ctx,
            @PathParam("workspace") String workspace,
            @PathParam("layername") String layerName,
            @PathParam("gfid") int gfid) {

        // use legacy response because some clients may still access old response type
        FeatureImageResponseLegacy response = new FeatureImageResponseLegacy();

        try {
            User user = UserRepository.getUser(sec.getUserPrincipal().getName()).get();

            int layerId = DatasetRepository.getLayerId(workspace, layerName);
            List<String> images = imageAndDocumentService.getFeatureImages(layerId, gfid);
            List<FeatureDocument> imagesWithDescription = imageAndDocumentService.getFeatureImagesOrDocuments(user,
                    workspace, layerId, gfid, DocumentType.IMAGE);

            response.setImages(images);
            response.setImagesWithDescription(imagesWithDescription);

        } catch (SQLException | InternalError e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        } catch (IOException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).entity("Could not retrieve image names. ").build();
        }

        return Response.ok(response).build();
    }

    @GET
    @Path("/{filename}")
    @Produces({ "image/png", "image/jpg", "image/jpeg", "image/gif" })
    @Operation(summary = "Retrieve an image of a feature by its name")
    public Response getFeatureImage(
            @Context SecurityContext sec,
            @Context ServletContext ctx,
            @PathParam("workspace") String workspace,
            @PathParam("layername") String layerName,
            @PathParam("gfid") int gfid,
            @PathParam("filename") String imageName) {

        try {
            int layerId = DatasetRepository.getLayerId(workspace, layerName);
            byte[] imageData = imageAndDocumentService.getFeatureImageOrDocument(ctx, workspace, layerId, gfid,
                    imageName, DocumentType.IMAGE);

            return Response.ok(new ByteArrayInputStream(imageData)).build();

        } catch (NoSuchFileException | NotFoundException e) {
            return Response.status(Status.NOT_FOUND).build();
        } catch (SQLException | InternalError e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        } catch (IOException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).entity("Could not retrieve image. ").build();
        }

    }
}
