package de.narimo.georepo.server.api;

import java.io.IOException;
import java.util.Arrays;
import java.util.List;

import javax.servlet.ServletContext;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.FormParam;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.SecurityContext;
import javax.ws.rs.ext.Provider;

import de.narimo.commons.json.JsonConverter;
import de.narimo.georepo.server.GeorepoConstants;
import de.narimo.georepo.server.GeorepoDatasource;
import de.narimo.georepo.server.GeorepoTools;
import de.narimo.georepo.server.dto.FeatureType;
import de.narimo.georepo.server.geoserver.GeoserverLayer;
import de.narimo.georepo.server.geoserver.GeoserverWorkspace;
import de.narimo.georepo.server.layer.GeometryType;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;

/**
 * georepo.com layer API
 *
 * @author Ulrich Mann
 *
 */
@Provider
@Path("/workspaces/{workspace}/layers")
@Tag(name = "Layers")
public class LayerController {

    @GET
    @Path("/")
    @Produces(MediaType.APPLICATION_JSON)
    @Operation(summary = "Retrieve workspace layers")
    public static Response getWorkspaceLayers(
            @Context SecurityContext sec,
            @Context ServletContext ctx,
            @PathParam("workspace") String workspace) {

        try {
            GeoserverLayer layer = new GeoserverLayer(ctx);
            GeoserverWorkspace gworkspace = new GeoserverWorkspace(layer.getGeoserverRestUrl(), layer.getHeaders(),
                    layer.getGeoserverUser(), layer.getGeoserverPass());

            List<FeatureType> layers = gworkspace.getWorkspaceFeatureTypes(workspace, false);

            return Response.ok().entity(JsonConverter.toJson(layers)).build();
        } catch (Exception e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }

    /**
     * Prerequisites for csv import: - only point data supported yet - column lon
     * for longitude coordinates - column lat for latitude coordinates - filename
     * must only contain alphanumerics, dashes and underscores - filename must end
     * .csv (postgres insert function restriction) - only allowed csvdelimiter is
     * semicolon (;) - currently only utf8 charset allowed, i.e., no umlauts, ß,
     * etc. in csv content - numeric values in csv are inserted as text to database
     *
     * TODO: check: do NOT allow feature type names containing whitespaces! This
     * will mess up queries that are not correctly url encoded or even then.
     *
     * NB: Conventions for new layers, datasources and workspaces: - datasource
     * starts with prefix gd_ and concatenates workspace name - layer name (not
     * layer title!) must not contain whitespaces and end with _gd_ plus the newly
     * created layerId
     *
     * @param sec
     * @param ctxt
     * @param workspace
     * @param sridAsString
     * @param delimiter
     * @param fileName
     * @param readableLayerName
     * @param fileType
     * @param optColumnTypes
     * @return
     * @throws Exception
     */
    public static String createLayer(
            ServletContext ctx,

            String workspace,

            // EPSG code, defaults to4326
            int srid,

            // necessary for geocsv, defaults to ;
            String delimiter,

            // should be the name as how it lives in geoserver_data_dir
            String fileName,
            String readableLayerName,
            String fileType,
            String optColumnHeaders,
            String optColumnTypes

    ) throws Exception {

        System.out.println("Column types to use: " + optColumnTypes);

        String[] columnTypes = optColumnTypes == null ? null : optColumnTypes.split("\\s*,\\s*");
        if (fileType.equals("geocsv")) {
            return GeorepoTools.importGeoCSV(
                    ctx, ctx.getInitParameter(GeorepoConstants.geoserverDataDir), srid, delimiter, fileName,
                    readableLayerName, workspace, columnTypes);
        } else {
            throw new IllegalArgumentException("Parameter filetype must specify a supported filetype.");
        }

    }

    @POST
    @Path("")
    @Consumes(MediaType.APPLICATION_FORM_URLENCODED)
    @Operation(summary = "Create a new empty layer", description = "Creates a new empty layer of the specified type.")
    public static Response createEmptyLayer(
            @Context ServletContext ctx,

            @PathParam("workspace") String workspace,
            @FormParam("layertitle") String layerTitle,
            @FormParam("geometrytype") String geometryType,
            @FormParam("columnheaders") String columnHeaders,

            // optional, comma-separated string with postgres column types, e.g.
            // real,text,text,integer...
            @FormParam("columntypes") String columnTypes) throws Exception {

        if (workspace == null) {
            throw new IOException("Workspace not defined.");
        }
        if (layerTitle == null) {
            throw new IllegalArgumentException("Missing or wrong parameter layertitle.");
        }
        if (columnTypes == null) {
            throw new IllegalArgumentException("Missing or wrong parameter columntypes.");
        }

        int srid = 4326;
        System.out.println("Column types to use: " + columnTypes);

        if (!Arrays.asList(GeometryType.values()).contains(GeometryType.valueOf(geometryType))) {
            throw new IOException("Invalid layer type. Must be one of POINT, LINESTRING or POLYGON.");
        }

        String[] columnTyps = columnTypes == null ? null : columnTypes.split("\\s*,\\s*");
        String[] columnHeadrs = columnHeaders == null ? null : columnHeaders.split("\\s*,\\s*");

        GeorepoTools.importEmptyLayer(ctx, srid, layerTitle, workspace, columnHeadrs,
                columnTyps, GeometryType.valueOf(geometryType));

        return Response.status(201).build();
    }

    @PUT
    @Path("/{layername}")
    @Operation(summary = "Update an existing layer")
    public static Response updateLayerContent(
            @Context ServletContext ctx,

            @PathParam("workspace") String workspace,
            @PathParam("layername") String layerName,

            @QueryParam("delimiter") String delimiter, // necessary for geocsv,
                                                       // defaults to ;
            @QueryParam("filename") String fileName, // should be the name as
                                                     // how it lives in
                                                     // geoserver_data_dir,
                                                     // csv!!
            @QueryParam("filetype") String fileType,
            @QueryParam("columntypes") String optColumnTypes // optional
                                                             // comma-separated
                                                             // string with
                                                             // postgres column
                                                             // types, e.g.
                                                             // real,text,text,integer...
    ) throws Exception {

        int srid = 4326;
        Integer layerId = null;
        try {
            layerId = Integer.valueOf(layerName.substring(layerName.lastIndexOf("_") + 1));
        } catch (Exception e) {
            throw new IllegalArgumentException("Invalid layer name " + layerName + " given. Must be in format "
                    + GeorepoDatasource.georepoDatasourcePrefix + "xxxx.");
        }

        if (fileType.equals("geocsv")) {
            return GeorepoTools.replaceGeoCSVLayer(
                    ctx,
                    ctx.getInitParameter(GeorepoConstants.geoserverDataDir),
                    srid,
                    delimiter,
                    fileName,
                    workspace,
                    optColumnTypes,
                    layerId);
        } else {
            throw new IllegalArgumentException("Parameter filetype is not a supported filetype: " + fileType);
        }

    }

    @DELETE
    @Path("/{layername}")
    @Operation(summary = "Remove a layer from the workspace.")
    public static Response removeLayer(
            @PathParam("workspace") String workspace,
            @PathParam("layername") String layername) {

        // TODO

        return Response.status(Status.NOT_IMPLEMENTED).build();
    }

    public static void importFile0(ServletContext ctx, String workspace, String fileName,
            String layerName, String optColumnTypes) throws Exception {

        int srid = 4326; // currently expected to be wgs84
        String csvDelimiter = ";";

        String fileType = "";
        String fileExt = fileName.substring(fileName.lastIndexOf(".") + 1);
        System.out.println("importing: " + fileName);
        System.out.println("fileextension: " + fileExt);
        if (fileExt.equals("csv")) {
            fileType = "geocsv";
        }

        LayerController.createLayer(ctx, workspace, srid, csvDelimiter, fileName, layerName, fileType, null,
                optColumnTypes);

    }
}
