package de.narimo.georepo.server.api;

import java.io.IOException;
import java.sql.SQLException;

import javax.servlet.ServletContext;
import javax.ws.rs.Consumes;
import javax.ws.rs.ForbiddenException;
import javax.ws.rs.NotFoundException;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.SecurityContext;
import javax.ws.rs.ext.Provider;

import de.narimo.commons.dto.User;
import de.narimo.geocore.ws.repository.UserRepository;
import de.narimo.georepo.server.api.comments.CommentOrLinkService;
import de.narimo.georepo.server.api.comments.CommentOrLinkType;
import de.narimo.georepo.server.api.comments.Weblink;
import de.narimo.georepo.server.repository.DatasetRepository;
import de.narimo.georepo.server.repository.DifftableRepository;
import de.narimo.georepo.server.tools.AdminTools;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;

@Provider
@Path("/workspaces/{diffWorkspace}/layers/{layername}/features/{gfid}/links")
@Tag(name = "Weblinks")
public class WeblinkModificationController {

    CommentOrLinkService commentService = new CommentOrLinkService();

    @POST
    @Path("/changes")
    @Consumes({ MediaType.APPLICATION_JSON })
    @Operation(summary = "Suggest adding a new web link to a feature of a layer")
    public Response postLayerWeblink(
            @Context SecurityContext sec,
            @Context ServletContext ctx,
            @PathParam("diffWorkspace") String diffWorkspace,
            @PathParam("layername") String dataLayerName,
            @PathParam("gfid") int gfid,

            Weblink weblink) throws IOException, SQLException {

        try {

            User user = UserRepository.getUser(sec.getUserPrincipal().getName()).get();

            String workspace = DifftableRepository.getDataWorkspace(diffWorkspace, dataLayerName);
            if (workspace == null) {
                throw new InternalError("No data workspace defined.");
            }

            // If user has no write permission on that workspace, he would already have been
            // thrown out by authorisation layer
            int layerId = DatasetRepository.getLayerId(workspace, dataLayerName);

            int commentId = commentService.addCommentOrWeblink(ctx, user, workspace, layerId, gfid, weblink,
                    CommentOrLinkType.WEBLINK);

        } catch (SQLException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }

        return Response.status(201).build();
    }

    @PUT
    @Path("/{linkId}/accept")
    @Operation(summary = "Accept a suggested web link")
    public Response acceptWeblinkChangeset(
            @PathParam("diffWorkspace") String diffWorkspace,
            @PathParam("layername") String dataLayerName,
            @PathParam("gfid") int gfid,
            @PathParam("linkId") int linkId,
            @Context SecurityContext sec) {

        try {
            User user = UserRepository.getUser(sec.getUserPrincipal().getName()).get();

            String workspace = DifftableRepository.getDataWorkspace(diffWorkspace, dataLayerName);
            if (workspace == null) {
                throw new InternalError("No data workspace defined.");
            }

            AdminTools.checkAdminPermission(user.getId(), workspace);

            int layerId = DatasetRepository.getLayerId(workspace, dataLayerName);
            commentService.acceptCommentOrWeblink(layerId, gfid, linkId);
            return Response.ok().build();

        } catch (NumberFormatException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).entity("Invalid feature id.").build();
        } catch (NotFoundException e) {
            e.printStackTrace();
            return Response.status(Status.NOT_FOUND).build();
        } catch (ForbiddenException e) {
            e.printStackTrace();
            return Response.status(Status.FORBIDDEN).build();
        } catch (InternalError | RuntimeException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }

    @PUT
    @Path("/{linkId}/decline")
    @Operation(summary = "Decline a suggested web link")
    public Response declineWeblinkChangeset(
            @PathParam("diffWorkspace") String diffWorkspace,
            @PathParam("layername") String dataLayerName,
            @PathParam("gfid") int gfid,
            @PathParam("linkId") int linkId,
            @Context SecurityContext sec) {

        try {
            User user = UserRepository.getUser(sec.getUserPrincipal().getName()).get();

            String workspace = DifftableRepository.getDataWorkspace(diffWorkspace, dataLayerName);
            if (workspace == null) {
                throw new InternalError("No data workspace defined.");
            }

            AdminTools.checkAdminPermission(user.getId(), workspace);

            int layerId = DatasetRepository.getLayerId(workspace, dataLayerName);
            commentService.declineCommentOrWeblink(layerId, gfid, linkId);
            return Response.ok().build();

        } catch (NumberFormatException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).entity("Invalid feature id.").build();
        } catch (NotFoundException e) {
            e.printStackTrace();
            return Response.status(Status.NOT_FOUND).build();
        } catch (ForbiddenException e) {
            e.printStackTrace();
            return Response.status(Status.FORBIDDEN).build();
        } catch (InternalError | RuntimeException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }
}
