package de.narimo.georepo.server.api.comments;

import java.sql.SQLException;
import java.util.Arrays;
import java.util.List;

import javax.servlet.ServletContext;
import javax.ws.rs.NotFoundException;

import de.narimo.commons.dto.User;
import de.narimo.georepo.server.repository.CommentRepository;
import de.narimo.georepo.server.repository.DatasetRepository;
import de.narimo.georepo.server.tools.TableTools;

public class CommentOrLinkService {

    public List<Weblink> getComments(User user, String workspace, int layerId, int gfid)
            throws SQLException {

        // if no comments table exists, return empty list
        boolean commentsTableExists = CommentRepository.checkCommentsTableExists(layerId);
        if (!commentsTableExists) {
            System.out.println("Comments table does not yet exist for layer " + layerId + ". No comments present.");
            return Arrays.asList();
        }

        return CommentRepository.getFeatureComments(user, workspace, layerId, gfid, CommentOrLinkType.COMMENT);
    }

    public List<Weblink> getDiffComments(User user, String workspace, int layerId)
            throws SQLException {

        // if no comments table exists, return empty list
        boolean commentsTableExists = CommentRepository.checkCommentsTableExists(layerId);
        if (!commentsTableExists) {
            System.out.println("Comments table does not yet exist for layer " + layerId + ". No comments present.");
            return Arrays.asList();
        }

        return CommentRepository.getDiffFeatureComments(user, workspace, layerId, CommentOrLinkType.COMMENT);
    }

    public List<Weblink> getWeblinks(User user, String workspace, int layerId, int gfid)
            throws SQLException {

        // if no comments table exists, return empty list
        boolean commentsTableExists = CommentRepository.checkCommentsTableExists(layerId);
        if (!commentsTableExists) {
            System.out.println("Weblinks table does not yet exist for layer " + layerId + ". No weblinks present.");
            return Arrays.asList();
        }

        return CommentRepository.getFeatureWeblinks(user, workspace, layerId, gfid);
    }

    public List<Weblink> getDiffWeblinks(User user, String workspace, int layerId)
            throws SQLException {

        // if no comments table exists, return empty list
        boolean commentsTableExists = CommentRepository.checkCommentsDiffTableExists(layerId);
        if (!commentsTableExists) {
            System.out.println(
                    "Weblinks table does not yet exist for layer " + layerId + ". No suggested weblinks present.");
            return Arrays.asList();
        }

        return CommentRepository.getFeatureDiffWeblinks(user, workspace, layerId);
    }

    public int addCommentOrWeblink(ServletContext ctx, User user, String workspace, int layerId, int gfid,
            CommentOrWeblink commentOrWeblink, CommentOrLinkType commentOrLinkType)
            throws SQLException {

        if (!DatasetRepository.doesFeatureExist(layerId, gfid)) {
            throw new NotFoundException();
        }

        boolean commentTableExists = CommentRepository.checkCommentsTableExists(layerId);
        if (!commentTableExists) {
            System.out.println("Comment table does not yet exist for layer " + layerId + ". Creating it...");
            CommentRepository.createCommentsTable(layerId);
        }

        boolean commentDiffTableExists = CommentRepository.checkCommentDiffTableExists(layerId);
        if (!commentDiffTableExists) {
            System.out.println("Comment diff table does not yet exist for layer " + layerId + ". Creating it...");
            CommentRepository.createCommentDiffTable(layerId);
        }

        // TODO: Do we need to do more checks on the comments validity?
        if (!isCommentOrLinkValid(
                commentOrLinkType == CommentOrLinkType.COMMENT ? ((Comment) commentOrWeblink).getComment()
                        : ((Weblink) commentOrWeblink).getLink(),
                commentOrLinkType)) {
            throw new IllegalArgumentException("Comment contains invalid characters.");
        }

        String text = (commentOrLinkType == CommentOrLinkType.COMMENT)
                ? ((Comment) commentOrWeblink).getComment()
                : ((Weblink) commentOrWeblink).getLink();

        String description = (commentOrLinkType == CommentOrLinkType.WEBLINK)
                ? ((Weblink) commentOrWeblink).getDescription()
                : null;

        return CommentRepository.insertUnreviewedCommentOrWeblink(layerId, user.getId(), gfid, text, commentOrLinkType,
                description);

//        inform the admins that a new comment has been uploaded to the workspace

//        boolean notifyAdmin = true;
//        if (notifyAdmin) {
//            Notifier.notifyWorkspaceAdminsAboutPOIChanges(ctx, dataWorkspace, dataLayerName);
//        }

    }

    public void acceptCommentOrWeblink(int layerId, int gfid, int commentId) {
        String commentsTable = TableTools.getCommentTableName(layerId);
        String diffCommentsTable = TableTools.getCommentDiffTableName(layerId);
        CommentRepository.acceptCommentOrWeblink(commentsTable, diffCommentsTable, gfid, commentId);
    }

    public void declineCommentOrWeblink(int layerId, int gfid, int commentId) {
        String diffCommentsTable = TableTools.getCommentDiffTableName(layerId);
        CommentRepository.declineCommentOrWeblink(diffCommentsTable, gfid, commentId);
    }

    private boolean isCommentOrLinkValid(String commentOrLink, CommentOrLinkType commentOrLinkType) {
        if (commentOrLinkType == CommentOrLinkType.COMMENT) {
            if (!(commentOrLink.contains("<") || commentOrLink.contains("script")
                    || commentOrLink.contains("iframe"))) {
                return true;
            }
        } else if (commentOrLinkType == CommentOrLinkType.WEBLINK) {
            if (!(commentOrLink.contains("<") || commentOrLink.contains("script")
                    || commentOrLink.contains("iframe"))) {
                return true;
            }
        }
        return false;
    }

    public void deleteCommentOrWeblink(int layerId, int gfid, int commentOrWeblinkId) {
        String commentsTable = TableTools.getCommentTableName(layerId);
        CommentRepository.deleteCommentOrWeblink(commentsTable, gfid, commentOrWeblinkId);
    }
}
