package de.narimo.georepo.server.layer;

import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;
import java.util.List;
import java.util.Map;

import org.geotools.feature.DefaultFeatureCollection;
import org.geotools.feature.FeatureCollection;
import org.geotools.feature.simple.SimpleFeatureBuilder;
import org.geotools.feature.simple.SimpleFeatureTypeBuilder;
import org.geotools.geojson.feature.FeatureJSON;
import org.geotools.geojson.geom.GeometryJSON;
import org.geotools.referencing.crs.DefaultGeographicCRS;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.Point;
import org.locationtech.jts.geom.Polygon;
import org.opengis.feature.simple.SimpleFeature;
import org.opengis.feature.simple.SimpleFeatureType;

public class FeatureBuilder {

    final SimpleFeatureType featureType;

    public FeatureBuilder(GeometryType geometryType, Map<String, Class> propertyTypes) {
        this.featureType = createFeatureType(geometryType, propertyTypes);
    }

    private static SimpleFeatureType createPointFeatureType(Map<String, Class> propertyTypes) {
        SimpleFeatureTypeBuilder builder = new SimpleFeatureTypeBuilder();
        builder.setName("geometry");
        builder.setCRS(DefaultGeographicCRS.WGS84);
        builder.setDefaultGeometry("geometry");

        builder.add("geometry", Point.class);
        for (String key : propertyTypes.keySet()) {
            builder.length(255).add(key, propertyTypes.get(key));
        }
        return builder.buildFeatureType();
    }

    private static SimpleFeatureType createFeatureType(GeometryType geometryType, Map<String, Class> propertyTypes) {
//        if (geometryType == null) {
//            throw new IllegalArgumentException("Invalid geometryType null.");
//        }

        SimpleFeatureTypeBuilder builder = new SimpleFeatureTypeBuilder();
        builder.setName("geometry");
        if (geometryType != null) {
            builder.setCRS(DefaultGeographicCRS.WGS84);
            builder.setDefaultGeometry("geometry");

            if (geometryType.equals(GeometryType.POINT)) {
                builder.add("geometry", Point.class);
            } else if (geometryType.equals(GeometryType.LINESTRING)) {
                builder.add("geometry", LineString.class);
            } else if (geometryType.equals(GeometryType.POLYGON)) {
                builder.add("geometry", Polygon.class);
            }
        }

        for (String key : propertyTypes.keySet()) {
            builder.length(255).add(key, propertyTypes.get(key));
        }
        return builder.buildFeatureType();
    }

    public SimpleFeature createSimpleFeature(
            Geometry geometry,
            Map<String, Object> featureProperties) {

        SimpleFeatureBuilder featureBuilder = new SimpleFeatureBuilder(this.featureType);
        if (geometry != null) {
            featureBuilder.add(geometry);
        }

        SimpleFeature simpleFeature = featureBuilder.buildFeature(null);

        for (String key : featureProperties.keySet()) {
            simpleFeature.setAttribute(key, featureProperties.get(key));
        }
        return simpleFeature;
    }

    public FeatureCollection createFeatureCollection(
            List<SimpleFeature> simpleFeatures) {

        FeatureCollection featureCollection = new DefaultFeatureCollection();

        for (SimpleFeature simpleFeature : simpleFeatures) {
            ((DefaultFeatureCollection) featureCollection).add(simpleFeature);
        }
        return featureCollection;
    }

    /**
     * Writes a feature collection to json, including null values.
     * 
     * @param featureCollection
     * @return
     * @throws IOException
     */
    public static String featureCollectionToJson(FeatureCollection featureCollection) throws IOException {

        boolean encodeNullValues = false;

        Writer output = new StringWriter();
        GeometryJSON gjson = new GeometryJSON(8);
        FeatureJSON fjson = new FeatureJSON(gjson);

        fjson.setEncodeNullValues(encodeNullValues);

        String jsonFeatureCollection = "{\"type\": \"FeatureCollection\", \"features\": []}";
        if (featureCollection != null && featureCollection.features().hasNext()) {
            fjson.writeFeatureCollection(featureCollection, output);
            jsonFeatureCollection = output.toString();
        }
        return jsonFeatureCollection;
    }
}
