package de.narimo.georepo.server.notification;

import java.io.IOException;
import java.util.List;

import javax.servlet.ServletContext;

import de.narimo.commons.dto.RegistrationLanguage;
import de.narimo.commons.dto.User;
import de.narimo.commons.ws.mail.Mailer;
import de.narimo.geocore.ws.registration.PasswordForgottenDetails;
import de.narimo.geocore.ws.registration.UserRegistrationDetails;
import de.narimo.georepo.server.tools.AdminTools;
import de.narimo.georepo.server.tools.Translator;

public class Notifier {

    private Translator t = new Translator();
    private Mailer mailer;
    private String mailerUser;
    private String mailerPass;
    private String sender = "info@georepo.com";

    public Notifier(ServletContext ctx) {
        mailer = new Mailer();
        mailerUser = ctx.getInitParameter("info_mail_user");
        mailerPass = ctx.getInitParameter("info_mail_pw");
    }

    /**
     * Send notification to workspace admins that POI changes have been submitted.
     */
    public static void notifyWorkspaceAdminsAboutPOIChanges(ServletContext ctx, String workspace,
            String dataLayerName) {

        Thread thread = new Thread("notifyAdminsAboutPOIChanges") {
            @Override
            public void run() {
                List<User> admins = AdminTools.getAdminUsers(workspace);
                Notifier notifier = new Notifier(ctx);
                for (User admin : admins) {
                    notifier.sendPoiUpdateAdminNotification(dataLayerName, admin.getEmail(), admin.getLanguage());
                }
            }
        };
        thread.start();
    }

    /**
     * Send notification to workspace admins that a new user has requested
     * registration.
     */
    public static void notifyWorkspaceAdminsAboutRegistration(ServletContext ctx, String workspace,
            String userEmail) {

        Thread thread = new Thread("notifyAdminsAboutRegistration") {
            @Override
            public void run() {
                List<User> admins = AdminTools.getAdminUsers(workspace);
                Notifier notifier = new Notifier(ctx);
                for (User admin : admins) {
                    notifier.notifyRecipientAboutWorkspaceRegistration(workspace, userEmail,
                            admin.getEmail(), admin.getLanguage());
                }
            }
        };
        thread.start();
    }

    /**
     * Notify a user about his successful registration.
     */
    public void notifyUserAboutRegistrationConfirmationLink(UserRegistrationDetails registrationDetails,
            String registrationLink, RegistrationLanguage language) {

        Thread thread = new Thread("notifyRegistration") {
            @Override
            public void run() {

                String defaultApplication = "georepo";
                String registeredApplication = registrationDetails.getClientApp() != null
                        && !registrationDetails.getClientApp().isEmpty() ? registrationDetails.getClientApp()
                                : defaultApplication;

                String subject = t.translate("TRANSLATION_NEW_USER_REGISTRATION_SUBJECT", language,
                        registeredApplication);
                String msgBody = t.translate("TRANSLATION_NEW_USER_REGISTRATION", language, registrationLink);
                String msgTeam = t.translate("TRANSLATION_TEAM_SIGNATURE", language, defaultApplication);

                mailer.sendMail(sender, registrationDetails.getEmail(), subject, msgBody + msgTeam, mailerUser,
                        mailerPass);
            }
        };
        thread.start();
    }

    /**
     * Notify narimo that a registration has been requested.
     *
     * @param ctx
     * @param registrationDetails
     */
    public void notifyNarimoAboutRegistration(UserRegistrationDetails registrationDetails,
            RegistrationLanguage language) {

        Thread thread = new Thread("notifyRecipientAboutRegistration") {
            @Override
            public void run() {

                String defaultApplication = "georepo";
                String registeredApplication = registrationDetails.getClientApp() != null
                        && !registrationDetails.getClientApp().isEmpty() ? registrationDetails.getClientApp()
                                : defaultApplication;

                String subject = t.translate("TRANSLATION_NEW_USER_REGISTRATION_NARIMO_SUBJECT", language,
                        registeredApplication);
                String msgBody = t.translate("TRANSLATION_NEW_USER_REGISTRATION_NARIMO", language,
                        registrationDetails.getEmail());
                String msgTeam = t.translate("TRANSLATION_TEAM_SIGNATURE", language, defaultApplication);

                mailer.sendMail(sender, "info@narimo.de", subject, msgBody + msgTeam, mailerUser, mailerPass);
            }
        };
        thread.start();
    }

    private void notifyRecipientAboutWorkspaceRegistration(
            String workspace, String registeredUserEmail, String recipientEmail,
            RegistrationLanguage language) {

        String subject = t.translate("TRANSLATION_NEW_USER_WORKSPACE_REGISTRATION_SUBJECT", language);
        String msgBody = t.translate("TRANSLATION_NEW_USER_WORKSPACE_REGISTRATION", language, recipientEmail,
                registeredUserEmail);
        String msgTeam = t.translate("TRANSLATION_TEAM_SIGNATURE", language, "georepo");

        mailer.sendMail(sender, recipientEmail, subject, msgBody + msgTeam, mailerUser, mailerPass);
    }

    /**
     * Send password reset mail when a password forgotten request was submitted.
     */
    public void notifyUserAboutPasswordResetLink(PasswordForgottenDetails passwordDetails,
            String resetLink, RegistrationLanguage language) {

        Thread thread = new Thread("notifyPasswordForgotten") {
            @Override
            public void run() {
                String defaultApplication = "georepo";
                String registeredApplication = passwordDetails.getClientApp() != null
                        && !passwordDetails.getClientApp().isEmpty() ? passwordDetails.getClientApp()
                                : defaultApplication;

                String subject = t.translate("TRANSLATION_PASSWORD_RESET_REQUEST_SUBJECT", language,
                        registeredApplication);
                String msgBody = t.translate("TRANSLATION_PASSWORD_RESET_REQUEST", language, resetLink);
                String msgTeam = t.translate("TRANSLATION_TEAM_SIGNATURE", language, defaultApplication);

                mailer.sendMail(sender, passwordDetails.getEmail(), subject, msgBody + msgTeam, mailerUser, mailerPass);
            }
        };
        thread.start();
    }

    public void notifyUserAboutSuccessfulPasswordReset(String recipientMail,
            RegistrationLanguage language) {

        Thread thread = new Thread("notifyPasswordResetSuccessful") {
            @Override
            public void run() {
                String defaultApplication = "georepo";

                String subject = t.translate("TRANSLATION_PASSWORD_RESET_SUBJECT", language,
                        defaultApplication);
                String msgBody = t.translate("TRANSLATION_PASSWORD_RESET", language);
                String msgTeam = t.translate("TRANSLATION_TEAM_SIGNATURE", language, defaultApplication);

                mailer.sendMail(sender, recipientMail, subject, msgBody + msgTeam, mailerUser, mailerPass);
            }
        };
        thread.start();
    }

    public void notifyUserAboutAccountDeletion(String recipientMail, String clientApp,
            RegistrationLanguage language) {

        Thread thread = new Thread("notifyAccountDeleteConfirmation") {
            @Override
            public void run() {
                String defaultApplication = "georepo";
                String registeredApplication = clientApp != null && !clientApp.isEmpty() ? clientApp
                        : defaultApplication;

                String subject = t.translate("TRANSLATION_ACCOUNT_DELETION_USER_HEADER", language,
                        registeredApplication);
                String msgBody = t.translate("TRANSLATION_ACCOUNT_DELETION_USER", language);
                String msgTeam = t.translate("TRANSLATION_TEAM_SIGNATURE", language, "georepo");

                mailer.sendMail(sender, recipientMail, subject, msgBody + msgTeam, mailerUser, mailerPass);
            }
        };
        thread.start();
    }

    public void notifyNarimoAboutAccountRemoval(String userAccountMail, String clientApp,
            RegistrationLanguage language) {

        Thread thread = new Thread("notifyNarimoAboutAccountDeletion") {
            @Override
            public void run() {
                String defaultApplication = "georepo";
                String registeredApplication = clientApp != null && !clientApp.isEmpty() ? clientApp
                        : defaultApplication;

                String subject = t.translate("TRANSLATION_ACCOUNT_DELETION_HEADER", language,
                        registeredApplication);
                String msgBody = t.translate("TRANSLATION_ACCOUNT_DELETION", language,
                        userAccountMail);
                String msgTeam = t.translate("TRANSLATION_TEAM_SIGNATURE", language, "georepo");

                mailer.sendMail(sender, "info@narimo.de", subject, msgBody + msgTeam, mailerUser, mailerPass);
            }
        };
        thread.start();
    }

    public void notifyNarimoAboutFailedAccountRemoval(String userAccountMail,
            String clientApp, RegistrationLanguage language) {

        Thread thread = new Thread("notifyNarimoAboutAccountDeletion") {
            @Override
            public void run() {
                String defaultApplication = "georepo";
                String registeredApplication = clientApp != null && !clientApp.isEmpty() ? clientApp
                        : defaultApplication;

                String subject = t.translate("TRANSLATION_FAILED_ACCOUNT_DELETION_HEADER", language,
                        registeredApplication);
                String msgBody = t.translate("TRANSLATION_FAILED_ACCOUNT_DELETION", language,
                        userAccountMail);
                String msgTeam = t.translate("TRANSLATION_TEAM_SIGNATURE", language,
                        defaultApplication);
                mailer.sendMail(sender, "info@narimo.de", subject, msgBody + msgTeam, mailerUser, mailerPass);
            }
        };
        thread.start();
    }

    /**
     * Send a notification that POI changes have been submitted.
     *
     * @param ctx
     * @param dataLayerName
     * @param recipient
     */
    private void sendPoiUpdateAdminNotification(String dataLayerName, String recipient,
            RegistrationLanguage language) {
        String subject = t.translate("TRANSLATION_SUGGESTED_DATASET_UPDATE_HEADER", language,
                "georepo");
        String msgBody = t.translate("TRANSLATION_SUGGESTED_DATASET_UPDATE", language, dataLayerName);
        String msgTeam = t.translate("TRANSLATION_TEAM_SIGNATURE", language, "georepo");
        mailer.sendMail(sender, recipient, subject, msgBody + msgTeam, mailerUser, mailerPass);
    }

    public void sendDeclinedAdminNotification(String dataLayerName, RegistrationLanguage language) {
        Thread thread = new Thread("updateDeclined") {
            @Override
            public void run() {
                String recipient = "u.mann@narimo.de";

                String subject = t.translate("TRANSLATION_DATASET_UPDATE_DECLINED_HEADER", language,
                        "georepo");
                String msgBody = t.translate("TRANSLATION_DATASET_UPDATE_DECLINED", language,
                        dataLayerName);
                String msgTeam = t.translate("TRANSLATION_TEAM_SIGNATURE", language, "georepo");
                mailer.sendMail(sender, recipient, subject, msgBody + msgTeam, mailerUser, mailerPass);
            }
        };
        thread.start();
    }

    public static void main(String[] args) throws IOException {

        Translator t = new Translator();
        System.out.println(t.translate("A", RegistrationLanguage.DE, "4", "some"));

    }
}