package de.narimo.georepo.server.repository;

import java.io.IOException;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.ws.rs.NotFoundException;

import org.geotools.feature.FeatureCollection;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.io.ParseException;
import org.opengis.feature.Property;
import org.opengis.feature.simple.SimpleFeature;

import de.narimo.commons.jdbc.JDBCConnectionJNDI;
import de.narimo.georepo.server.GeorepoConstants;
import de.narimo.georepo.server.dto.DiffDataset;
import de.narimo.georepo.server.tools.GeometryTools;
import de.narimo.georepo.server.tools.QueryCheck;
import de.narimo.georepo.server.tools.TableTools;

public class DiffDatasetRepository extends ParentDatasetRepository {

    /**
     * Insert new datasets into a diff table.
     *
     * @param tableName the name of a diff table
     * @param features
     */
    public static void insertDatasets(String diffTableName, String diffAuthTableName, List<SimpleFeature> features,
            long userId) throws IOException {

        QueryCheck.checkTableName(diffTableName);
        QueryCheck.checkDiffTableName(diffTableName);
        QueryCheck.checkTableName(diffAuthTableName);

        JDBCConnectionJNDI jdbcData = null;

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            Map<String, String> columnsAndTypes = getColumnTypes(diffTableName, GeorepoConstants.DEFAULTSCHEMA);
            columnsAndTypes.put(GeorepoConstants.GRPSTATUSCOLUMN, "text");
            columnsAndTypes.put(GeorepoConstants.GEOMETRYCOLUMN, "geom");

            String insert = createFeatureInsertUpdateString(diffTableName, columnsAndTypes, features, 'n');

            ResultSet rs = jdbcData.executeQuery(insert);

            List<Integer> insertedGrpfids = new ArrayList<>();
            while (rs.next()) {
                insertedGrpfids.add(rs.getInt(GeorepoConstants.GRPFIDCOLUMN));
            }

            String auth = createDiffAuthTableUpdateString(insertedGrpfids, userId, diffAuthTableName);
            jdbcData.execute(auth);
            if (!jdbcData.getAutoCommit()) {
                jdbcData.commit();
            }

        } catch (SQLException e) {
            e.printStackTrace();
            throw new IOException(e);
        } catch (Exception e) {
            e.printStackTrace();
            throw new IOException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    /**
     * Insert datasets into a diff table that declare updates from users.
     *
     * @param diffTableName
     * @param features
     */
    public static void updateDatasets(String diffTableName, String diffAuthTableName, List<SimpleFeature> features,
            long userId) throws IOException {

        QueryCheck.checkTableName(diffTableName);
        QueryCheck.checkDiffTableName(diffTableName);
        QueryCheck.checkTableName(diffAuthTableName);

        JDBCConnectionJNDI jdbcData = null;

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            Map<String, String> columnsAndTypes = getColumnTypes(diffTableName, GeorepoConstants.DEFAULTSCHEMA);
            columnsAndTypes.put(GeorepoConstants.GRPSTATUSCOLUMN, "text");
            columnsAndTypes.put(GeorepoConstants.GEOMETRYCOLUMN, "geom");

            String update = createFeatureInsertUpdateString(diffTableName, columnsAndTypes, features, 'm');

            ResultSet rs = jdbcData.executeQuery(update);

            List<Integer> insertedGrpfids = new ArrayList<>();
            while (rs.next()) {
                insertedGrpfids.add(rs.getInt(GeorepoConstants.GRPFIDCOLUMN));
            }

            String auth = createDiffAuthTableUpdateString(insertedGrpfids, userId, diffAuthTableName);
            jdbcData.execute(auth);
            if (!jdbcData.getAutoCommit()) {
                jdbcData.commit();
            }

        } catch (SQLException e) {
            e.printStackTrace();
            throw new IOException(e);
        } catch (Exception e) {
            e.printStackTrace();
            throw new IOException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    /**
     * Add diff datasets as delete entries.
     * 
     * @param diffTableName
     * @param diffAuthTableName
     * @param features
     * @param userId
     * @throws IOException
     */
    public static void deleteDatasets(String diffTableName, String diffAuthTableName, List<SimpleFeature> features,
            long userId) throws IOException {

        QueryCheck.checkTableName(diffTableName);
        QueryCheck.checkDiffTableName(diffTableName);
        QueryCheck.checkTableName(diffAuthTableName);

        JDBCConnectionJNDI jdbcData = null;

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            Map<String, String> columnsAndTypes = getColumnTypes(diffTableName, GeorepoConstants.DEFAULTSCHEMA);
            columnsAndTypes.put(GeorepoConstants.GRPSTATUSCOLUMN, "text");

            String delete = createFeatureDeleteString(diffTableName, features);
            ResultSet rs = jdbcData.executeQuery(delete);

            List<Integer> insertedGrpfids = new ArrayList<>();
            while (rs.next()) {
                insertedGrpfids.add(rs.getInt(GeorepoConstants.GRPFIDCOLUMN));
            }

            String auth = createDiffAuthTableUpdateString(insertedGrpfids, userId, diffAuthTableName);
            jdbcData.execute(auth);
            if (!jdbcData.getAutoCommit()) {
                jdbcData.commit();
            }

        } catch (SQLException e) {
            e.printStackTrace();
            throw new IOException(e);
        } catch (Exception e) {
            e.printStackTrace();
            throw new IOException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    /**
     * Add, update or remove a dataset from an original data table.
     *
     * @param diffTable
     * @param dataTable
     * @param grpfid
     * @param features
     * @param userId
     * @throws IOException
     */
    public static void acceptDataset(int layerId, int grpfid)
            throws IOException {

        JDBCConnectionJNDI jdbcData = null;

        try {

            try {
                jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");
            } catch (Exception e) {
                throw new SQLException(e);
            }

            String dataTable = TableTools.getDataTableName(layerId);
            String diffTable = TableTools.getDiffTableName(layerId);
            String diffAuthTable = TableTools.getDiffAuthTableName(layerId);

            // create the auth table that holds user and timestamp data for the gd_x table
            // but only if there is none yet
            String authTable = TableTools.getAuthTableName(layerId);
            DatasetRepository.createAuthTable(authTable);

            Map<String, String> columnsAndTypes = getColumnTypes(diffTable, GeorepoConstants.DEFAULTSCHEMA);
            columnsAndTypes.put(GeorepoConstants.GEOMETRYCOLUMN, "geom");

            List<String> columns = new ArrayList<>(columnsAndTypes.keySet());
            String accept = createAcceptFeatureString(layerId, dataTable, authTable, diffTable, diffAuthTable, grpfid,
                    columns);
            System.out.println(accept);

            try {
                jdbcData.execute(accept);
                if (!jdbcData.getAutoCommit()) {
                    jdbcData.commit();
                }
            } catch (Exception e) {
                throw new SQLException(e);
            }

        } catch (SQLException e) {
            e.printStackTrace();
            throw new IOException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    public static void declineDataset(String diffTable, String diffAuthTable, int grpfid) throws IOException {

        QueryCheck.checkTableName(diffTable);
        QueryCheck.checkTableName(diffAuthTable);
        QueryCheck.checkDiffTableName(diffTable);

        JDBCConnectionJNDI jdbcData = null;
        try {

            try {
                jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");
            } catch (Exception e) {
                throw new SQLException(e);
            }

            String decline = createDeclineChangesetString(diffTable, diffAuthTable, grpfid);
            System.out.println(decline);

            try {
                jdbcData.execute(decline);
                if (!jdbcData.getAutoCommit()) {
                    jdbcData.commit();
                }
            } catch (Exception e) {
                throw new SQLException(e);
            }

        } catch (SQLException e) {
            e.printStackTrace();
            throw new IOException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    /**
     * Creates the SQL string to insert feature updates.
     *
     * @param tableName
     * @param columnsAndTypes
     * @param features
     * @param status
     * @return
     * @throws IOException
     * @throws ParseException
     */
    private static String createFeatureInsertUpdateString(String tableName, Map<String, String> columnsAndTypes,
            List<SimpleFeature> features, char status) throws IOException, ParseException {

        QueryCheck.checkTableName(tableName);

        if (!GeorepoConstants.ALLOWEDUPDATESTATI.contains(String.valueOf(status))) {
            throw new IllegalArgumentException("Unknown update status: " + status);
        }

        String columnNames = String.join(",", columnsAndTypes.keySet());

        StringBuilder b = new StringBuilder();
        b.append("INSERT INTO public." + tableName + "(" + columnNames + ") VALUES ");

        int j = 0;
        for (SimpleFeature feature : features) {

            // has property gfid to reference real world feature?
            if (status == 'm') {
                if (!hasValidGfid(feature)) {
                    throw new IllegalArgumentException(
                            "At least one modified feature is missing or has invalid required property '"
                                    + GeorepoConstants.GFIDCOLUMN + "'.");
                }
            } else if (status == 'n') {
                if (!gfidIsNull(feature)) {
                    throw new IllegalArgumentException(
                            "New features may not specify property '" + GeorepoConstants.GFIDCOLUMN + "'.");
                }
            }

            if (j > 0) {
                b.append(",");
            }
            b.append("(");
            int i = 0;

            Geometry geom = (Geometry) feature.getDefaultGeometry();

            for (String column : columnsAndTypes.keySet()) {
                if (i > 0) {
                    b.append(",");
                }
                if (column.equals(GeorepoConstants.GRPSTATUSCOLUMN)) {
                    b.append("'" + status + "'");

                } else if (column.equals(GeorepoConstants.GEOMETRYCOLUMN) && geom != null) {
                    String wkt = GeometryTools.createWKT(geom);
                    b.append("ST_GeomFromText('" + wkt + "'," + GeorepoConstants.allowedSRID + ")");

                } else {
                    Property p = feature.getProperty(column);
                    if (p != null && p.getValue() != null) {
                        String rawValue = p.getValue().toString();
                        if (columnsAndTypes.get(column).equals("text")
                                || columnsAndTypes.get(column).equals("character varying")) {
                            // character types
                            b.append("'" + escapeSingleQuote(rawValue) + "'");
                        } else if (columnsAndTypes.get(column).equals("boolean")) {
                            b.append(Boolean.valueOf(rawValue));
                        } else {
                            // numeric types
                            if (columnsAndTypes.get(column).equals("integer")) {
                                b.append(Integer.valueOf(rawValue));
                            } else {
                                b.append(Double.valueOf(rawValue));
                            }
                        }
                    } else {
                        b.append("null");
                    }
                }
                i++;
            }
            b.append(")");
            j++;
        }
        b.append(" RETURNING " + GeorepoConstants.GRPFIDCOLUMN + ";");

        System.out.println("Inserting datasets: " + b.toString());

        return b.toString();
    }

    static boolean hasValidGfid(SimpleFeature feature) {
        if (gfidIsNull(feature)) {
            return false;
        }
        try {
            Integer.valueOf(feature.getProperty(GeorepoConstants.GFIDCOLUMN).getValue().toString());
        } catch (NumberFormatException e) {
            e.printStackTrace();
            return false;
        }
        return true;
    }

    static boolean gfidIsNull(SimpleFeature feature) {
        if (feature.getProperty(GeorepoConstants.GFIDCOLUMN) == null
                || feature.getProperty(GeorepoConstants.GFIDCOLUMN).getValue() == null) {
            // mh, do we want to accept "null" as a String really as a null value??,
            // see GfidCheckTest
//            if (String.valueOf(feature.getProperty(GeorepoConstants.GFIDCOLUMN).getValue()).equals("null")) {
//                return false;
//            }
            return true;
        }
        return false;
    }

    /**
     * Creates a sql query string to insert a delete feature changeset to a diff
     * table.
     *
     * @param diffTableName
     * @param features
     * @return
     * @throws IOException
     * @throws ParseException
     */
    private static String createFeatureDeleteString(String diffTableName, List<SimpleFeature> features)
            throws IOException, ParseException {
        char status = 'd';

        QueryCheck.checkTableName(diffTableName);
        QueryCheck.checkDiffTableName(diffTableName);

        if (!GeorepoConstants.ALLOWEDUPDATESTATI.contains(String.valueOf(status))) {
            throw new IllegalArgumentException("Unknown update status: " + status);
        }

        StringBuilder b = new StringBuilder();
        b.append("INSERT INTO public." + diffTableName + "(" + GeorepoConstants.GFIDCOLUMN + ","
                + GeorepoConstants.GRPSTATUSCOLUMN + "," + GeorepoConstants.GEOMETRYCOLUMN + ") VALUES ");

        int j = 0;
        for (SimpleFeature feature : features) {

            // has property gfid to reference real world feature?
            if (!hasValidGfid(feature)) {
                throw new IllegalArgumentException(
                        "At least one delete feature is missing required property " + GeorepoConstants.GFIDCOLUMN
                                + ".");
            }

            if (j > 0) {
                b.append(",");
            }
            b.append("(");
            b.append("'" + feature.getProperty(GeorepoConstants.GFIDCOLUMN).getValue() + "'");
            b.append(",");
            b.append("'" + status + "'");

            b.append(",");

            Geometry geom = (Geometry) feature.getDefaultGeometry();
            if (geom != null) {
                String wkt = GeometryTools.createWKT(geom);
                b.append("ST_GeomFromText('" + wkt + "'," + GeorepoConstants.allowedSRID + ")");
            } else {
                // add NULL geometry for backwards compatibility (i.e. geometry on delete
                // request is optional)
                b.append("NULL");
            }

            b.append(")");
            j++;
        }
        b.append(" RETURNING " + GeorepoConstants.GRPFIDCOLUMN + ";");

        return b.toString();
    }

    /**
     * Creates a sql query string to update insert a new auth dataset.
     *
     * @param grpfids
     * @param userid
     * @param diffAuthTableName
     * @return
     * @throws IOException
     */
    private static String createDiffAuthTableUpdateString(List<Integer> grpfids, long userid, String diffAuthTableName)
            throws IOException {

        QueryCheck.checkTableName(diffAuthTableName);

        StringBuilder b = new StringBuilder();
        b.append("INSERT INTO public." + diffAuthTableName + " (" + GeorepoConstants.GRPFIDCOLUMN
                + ", insertedby) VALUES ");

        int i = 0;
        for (int grpfid : grpfids) {
            if (i > 0) {
                b.append(",");
            }
            b.append("(" + grpfid + "," + userid + ")");
            i++;
        }
        b.append(";");
        System.out.println(b.toString());
        return b.toString();
    }

    /**
     * Creates a sql query string to copy a changeset from a diff table to an
     * original data table.
     *
     * @param tableName
     * @param columnsAndTypes
     * @param features
     * @param status
     * @return
     * @throws SQLException when the columns differ from the original data table
     */
    private static String createAcceptFeatureString(int layerId, String dataTable, String authTable, String diffTable,
            String diffAuthTable,
            int grpfid, List<String> changesetColumns) throws NotFoundException, IllegalArgumentException, IOException {

        QueryCheck.checkTableName(dataTable);
        QueryCheck.checkTableName(authTable);
        QueryCheck.checkTableName(diffTable);
        QueryCheck.checkTableName(diffAuthTable);
        QueryCheck.checkDiffTableName(diffTable);

        DiffDataset diffDataset = getDiffDatasetProperties(diffTable, grpfid);
        if (diffDataset == null) {
            throw new NotFoundException("No changeset with id " + grpfid + " exists;");
        }
        char status = diffDataset.getGrpstatus().charAt(0);
        Integer gfid = diffDataset.getGfid();

        System.out.println("Changeset to merge: status: " + status + ", gfid: " + gfid);

        changesetColumns.remove(GeorepoConstants.GRPSTATUSCOLUMN);
        changesetColumns.remove(GeorepoConstants.GRPFIDCOLUMN);
        changesetColumns.remove(GeorepoConstants.GFIDCOLUMN);

        StringBuilder b = new StringBuilder();
        if (status == 'm' && gfid != null) {

            // backup current dataset to hist table
            b.append(createBackupFeatureString(layerId, gfid, changesetColumns));

            // update the data table entry with accepted changes
            b.append(createAcceptModifiedFeatureString(dataTable, authTable, diffTable, diffAuthTable, grpfid,
                    changesetColumns));

        } else if (status == 'n' && gfid == null) {
            // we don't need any backup here since we create a new dataset entry

            b.append(createAcceptNewFeatureString(dataTable, authTable, diffTable, diffAuthTable, grpfid,
                    changesetColumns));

        } else if (status == 'd' && gfid != null) {

            // backup current dataset to hist table
            b.append(createBackupFeatureString(layerId, gfid, changesetColumns));

            // update the data table entry with accepted delete change
            b.append(createAcceptDeleteFeatureString(dataTable, authTable, gfid));

        } else {
            throw new IllegalArgumentException("Unsupported status " + status + " or corrupt changeset.");
        }

        String cleanupDiffChangesetSql = " DELETE FROM public." + diffTable + " WHERE " + GeorepoConstants.GRPFIDCOLUMN
                + "=" + grpfid + ";";
        b.append(cleanupDiffChangesetSql);

        String cleanupAuthChangesetSql = " DELETE FROM public." + diffAuthTable + " WHERE "
                + GeorepoConstants.GRPFIDCOLUMN + "=" + grpfid + ";";
        b.append(cleanupAuthChangesetSql);

        if (status == 'd' && gfid != null) {
            // If we accept a DELETE changeset, we have to remove
            // all remaining changesets for the same gfid as well
            // to prevent inconsistencies
            String cleanupAllRelatedAuthChangesetsSql = " DELETE FROM public." + diffAuthTable + " a USING public."
                    + diffTable + " d WHERE a." + GeorepoConstants.GRPFIDCOLUMN + "=d." + GeorepoConstants.GRPFIDCOLUMN
                    + " AND d." + GeorepoConstants.GFIDCOLUMN + "=" + gfid + ";";
            b.append(cleanupAllRelatedAuthChangesetsSql);

            String cleanupAllRelatedDiffChangesetsSql = " DELETE FROM public." + diffTable + " WHERE "
                    + GeorepoConstants.GFIDCOLUMN + "=" + gfid + ";";
            b.append(cleanupAllRelatedDiffChangesetsSql);
        }
        return b.toString();
    }

    private static String createDeclineChangesetString(String diffTable, String diffAuthTable, int grpfid)
            throws NotFoundException, IllegalArgumentException, IOException {

        QueryCheck.checkTableName(diffTable);
        QueryCheck.checkTableName(diffAuthTable);
        QueryCheck.checkDiffTableName(diffTable);

        DiffDataset diff = getDiffDatasetProperties(diffTable, grpfid);
        if (diff == null) {
            throw new NotFoundException("No changeset with id " + grpfid + " exists;");
        }

        StringBuilder b = new StringBuilder();
        b.append("DELETE FROM public." + diffTable + " WHERE " + GeorepoConstants.GRPFIDCOLUMN + "=" + grpfid + ";");
        b.append(
                "DELETE FROM public." + diffAuthTable + " WHERE " + GeorepoConstants.GRPFIDCOLUMN + "=" + grpfid + ";");
        return b.toString();
    }

    private static String createAcceptModifiedFeatureString(String dataTable, String authTable, String diffTable,
            String diffAuthTable, int grpfid, List<String> changesetColumns)
            throws NotFoundException, IllegalArgumentException {

        StringBuilder b = new StringBuilder();

        b.append("UPDATE public." + dataTable + " o SET ");

        int c = 0;
        for (String column : changesetColumns) {
            if (column.equals(GeorepoConstants.GRPSTATUSCOLUMN) || column.equals(GeorepoConstants.GRPFIDCOLUMN)
                    || column.equals(GeorepoConstants.GFIDCOLUMN)) {
                continue;
            }
            if (c > 0) {
                b.append(", ");
            }

            // #281: do not overwrite with values that are null in the diff table!
            // that is, write the original value, if there is a null value
            b.append(column + "=COALESCE(d." + column + ", o." + column + ")");
            c++;
        }
        b.append(" FROM " + diffTable + " d WHERE o.gfid=d.gfid");
        b.append(" AND d.grpfid=" + grpfid + "; ");

        // update or insert auth dataset (update might fail for legacy datasets without
        // auth entry)
        b.append("INSERT INTO  public." + authTable + "(grpfid, modifiedby, modifiedat)"
                + " (SELECT o.grpfid, da.insertedby, da.insertedat"
                + "     FROM gd_34 o, gd_34_diff d, gd_34_diff_auth da"
                + "     WHERE o.gfid=d.gfid AND da.grpfid=d.grpfid AND d.grpfid=" + grpfid
                + " )"
                + " ON CONFLICT (grpfid)"
                + " DO"
                + " UPDATE SET modifiedby=EXCLUDED.modifiedby, modifiedat=EXCLUDED.modifiedat;");
        return b.toString();
    }

    private static String createAcceptNewFeatureString(String dataTable, String authTable, String diffTable,
            String diffAuthTable, int grpfid, List<String> changesetColumns)
            throws NotFoundException, IllegalArgumentException {

        StringBuilder b = new StringBuilder();
        String columnNames = String.join(",", changesetColumns);

        b.append("WITH ins1 AS ("
                + " INSERT INTO public." + dataTable + "(" + columnNames + ") "
                + " (SELECT " + columnNames + " FROM public." + diffTable
                + "     WHERE grpfid=" + grpfid + ")"
                + "     RETURNING grpfid"
                + " )"
                + " INSERT INTO public." + authTable + "(grpfid, modifiedby, modifiedat, diff_grpfid)"
                + " (SELECT grpfid, 0, '1970-01-01 00:00:01', " + grpfid
                + " FROM ins1"
                + " );");

        b.append("UPDATE public." + authTable + " a"
                + " SET modifiedby=da.insertedby, modifiedat=da.insertedat "
                + " FROM " + diffAuthTable + " da"
                + " WHERE da.grpfid=a.diff_grpfid and da.grpfid=" + grpfid + ";");

        return b.toString();
    }

    private static String createAcceptDeleteFeatureString(String dataTable, String authTable, int gfid)
            throws NotFoundException, IllegalArgumentException {

        QueryCheck.checkTableName(dataTable);
        QueryCheck.checkTableName(authTable);

        // gfid is unique in orignal data table
        StringBuilder b = new StringBuilder();
        b.append("DELETE FROM public." + authTable + " as b using public." + dataTable
                + " as a WHERE a." + GeorepoConstants.GRPFIDCOLUMN + "=b." + GeorepoConstants.GRPFIDCOLUMN
                + " and a." + GeorepoConstants.GFIDCOLUMN + "=" + gfid + ";");
        b.append("DELETE FROM public." + dataTable + " " + "WHERE " + GeorepoConstants.GFIDCOLUMN + "=" + gfid
                + ";");

        return b.toString();
    }

    private static DiffDataset getDiffDatasetProperties(String diffTable, int grpfid) throws IOException {

        QueryCheck.checkTableName(diffTable);
        QueryCheck.checkDiffTableName(diffTable);

        JDBCConnectionJNDI jdbcData = null;

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql = "SELECT * FROM public." + diffTable + " WHERE " + GeorepoConstants.GRPFIDCOLUMN + "=?;";

            PreparedStatement ps = jdbcData.prepareStatement(sql);
            ps.setInt(1, grpfid);

            ResultSet rs = ps.executeQuery();
            if (rs.next()) {
                DiffDataset diff = new DiffDataset();
                diff.setGfid(rs.getInt(GeorepoConstants.GFIDCOLUMN));
                if (rs.wasNull()) {
                    // result set returns 0 for integer null values
                    diff.setGfid(null);
                }
                diff.setGrpfid(rs.getInt(GeorepoConstants.GRPFIDCOLUMN));
                diff.setGrpstatus(rs.getString(GeorepoConstants.GRPSTATUSCOLUMN));
                return diff;
            }
        } catch (Exception e) {
            e.printStackTrace();
            throw new IOException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
        return null;
    }

    public static FeatureCollection getModifiedDatasets(String diffTableName, String diffAuthTableName, long userId,
            boolean asAdmin) throws IOException {

        QueryCheck.checkTableName(diffTableName);
        QueryCheck.checkTableName(diffAuthTableName);

        JDBCConnectionJNDI jdbcData = null;

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            Map<String, String> columnsAndTypes = getColumnTypes(diffTableName, GeorepoConstants.DEFAULTSCHEMA);
            columnsAndTypes.put(GeorepoConstants.GRPSTATUSCOLUMN, "text");
            columnsAndTypes.put(GeorepoConstants.GRPFIDCOLUMN, "integer");

            Map<String, Class> propertyTypes = new HashMap<String, Class>();

            for (String colName : columnsAndTypes.keySet()) {
                String colType = columnsAndTypes.get(colName);
                if (colType.equals("integer")) {
                    propertyTypes.put(colName, Integer.class);
                } else if (colType.equals("double precision")) {
                    propertyTypes.put(colName, Double.class);
                } else {
                    propertyTypes.put(colName, String.class);
                }
            }

            String query = getModifiedDatasetsQuery(diffTableName, diffAuthTableName, userId, asAdmin);
            return createSimpleFeatures(diffTableName, propertyTypes, columnsAndTypes, query, null, false);

        } catch (SQLException e) {
            e.printStackTrace();
            throw new IOException(e);
        } catch (Exception e) {
            throw new IOException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    /**
     * Returns the sql query string to use for retrieving modified features. If
     * asAdmin is true, the query will fetch all features from the diff table.
     * Otherwise it fetches only those datasets, that are created by the requesting
     * user.
     *
     * @param diffTableName
     * @param userId
     * @param asAdmin
     * @return
     * @throws IOException
     */
    private static String getModifiedDatasetsQuery(String diffTableName, String diffAuthTableName, long userId,
            boolean asAdmin) throws IOException {
        QueryCheck.checkTableName(diffTableName);
        QueryCheck.checkTableName(diffAuthTableName);

        if (asAdmin) {
            return "SELECT ST_AsText(the_geom), * from public." + diffTableName + ";";
        } else {
            return "SELECT ST_AsText(the_geom), * from public." + diffTableName + " d JOIN public."
                    + diffAuthTableName + " a " + "ON d." + GeorepoConstants.GRPFIDCOLUMN + " = a."
                    + GeorepoConstants.GRPFIDCOLUMN + " " + "WHERE a.insertedby=" + userId + ";";
        }
    }

    private static String escapeSingleQuote(String text) { return text.replace("'", "''"); }

    /**
     * Creates an SQL string that inserts a dataset to a history table and removed
     * the oldest entry from the same table with the same gfid as the inserted
     * dataset.
     * 
     * @param layerId
     * @param grpfid
     * @param changesetColumns
     * @return
     * @throws NotFoundException
     * @throws IllegalArgumentException
     */
    private static String createBackupFeatureString(int layerId, int gfid, List<String> changesetColumns)
            throws NotFoundException, IllegalArgumentException {

        String dataTable = TableTools.getDataTableName(layerId);
        String authTable = TableTools.getAuthTableName(layerId);
        String histTable = TableTools.getHistTableName(layerId);

        StringBuilder b = new StringBuilder();

        String sql = "CREATE TABLE IF NOT EXISTS public." + histTable
                + " (LIKE " + dataTable
                + " INCLUDING DEFAULTS INCLUDING CONSTRAINTS INCLUDING INDEXES);"
                + " ALTER TABLE " + histTable
                + " ADD COLUMN IF NOT EXISTS insertedat timestamp without time zone DEFAULT timezone('utc'::text, now()), "
                + " ADD COLUMN IF NOT EXISTS insertedby integer; ";
        b.append(sql);

        String columnNames = String.join(",", changesetColumns);
        String qualifiedColumnNames = "a." + String.join(",a.", changesetColumns);

        String sql2 = "INSERT INTO public." + histTable + "(" + columnNames + ", " + GeorepoConstants.GFIDCOLUMN + ", "
                + GeorepoConstants.INSERTEDBYCOLUMN + ", " + GeorepoConstants.INSERTEDATCOLUMN + ")"
                + " SELECT " + qualifiedColumnNames + ", a." + GeorepoConstants.GFIDCOLUMN
                + ", b.modifiedby, b.modifiedat"
                + " FROM public." + dataTable + " as a LEFT OUTER JOIN public." + authTable + " as b"
                + " ON a." + GeorepoConstants.GRPFIDCOLUMN + "=b." + GeorepoConstants.GRPFIDCOLUMN
                + " WHERE a." + GeorepoConstants.GFIDCOLUMN + "=" + gfid + ";";
        b.append(sql2);

        // keep the most recent 5 datasets for this gfid, remove the oldest.
        String sql3 = "DELETE FROM " + histTable + " WHERE grpfid = ("
                + " SELECT a.grpfid FROM ("
                + "  SELECT grpfid, insertedat FROM " + histTable + " WHERE " + GeorepoConstants.GFIDCOLUMN + " = "
                + gfid
                + "  ORDER BY insertedat DESC OFFSET 5) a "
                + " ORDER BY a.insertedat ASC LIMIT 1); ";
        b.append(sql3);

        return b.toString();
    }

}
