package de.narimo.georepo.server.repository;

import java.io.IOException;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import de.narimo.commons.jdbc.JDBCConnectionJNDI;
import de.narimo.georepo.server.GeorepoConstants;
import de.narimo.georepo.server.tools.QueryCheck;

public class DifftableRepository {

    /**
     * Retrieve a diff workspace for a data workspace, assuming there is a
     * one-to-one relation.
     *
     * A diff workspace holds tables with user generated, not reviewed data updates.
     * A data workspace holds the original data tables where updates should be
     * merged only after admin review.
     * 
     * @param dataWorkspace
     * @return
     */
    public static String getDiffWorkspace(String dataWorkspace) {
        JDBCConnectionJNDI jdbcMeta = null;

        try {
            jdbcMeta = new JDBCConnectionJNDI("jdbc/georepoMetaResource");

            // TODO: careful! This assumes, we haven't added inconsistencies to the
            // difftables table before!
            String sql1 = "SELECT distinct(diffworkspace) FROM difftables WHERE dataworkspace=?;";
            System.out.println(sql1);

            PreparedStatement ps1 = jdbcMeta.prepareStatement(sql1);
            ps1.setString(1, dataWorkspace);

            ResultSet rs1 = ps1.executeQuery();
            String diffworkspace = null;

            // Check there is only one entry!
            int rsCount = 0;
            while (rs1.next()) {
                diffworkspace = rs1.getString("diffworkspace");
                rsCount++;
            }

            if (rsCount > 1) {
                throw new InternalError("Diff workspace cannot be determined uniquely!"
                        + " Multiple combinations of diffworkspace and dataworkspace exist in difftables.");
            }

            if (diffworkspace == null) {
                // check if the given data workspace is a diff workspace
                // actually

                String sql2 = "SELECT count(*) FROM difftables WHERE diffworkspace=?";
                System.out.println(sql2);

                PreparedStatement ps2 = jdbcMeta.prepareStatement(sql2);
                ps2.setString(1, dataWorkspace);

                ResultSet rs2 = ps2.executeQuery();
                int diffWorkspaceCount = 0;
                if (rs2.next()) {
                    diffWorkspaceCount = rs2.getInt("count");
                }
                if (diffWorkspaceCount > 0) {
                    diffworkspace = dataWorkspace;
                }
            }
            System.out.println("Diff workspace for " + dataWorkspace + " is " + diffworkspace + ".");
            return diffworkspace;

        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (jdbcMeta != null) {
                jdbcMeta.closeAll();
                jdbcMeta = null;
            }
        }
        return null;
    }

    /**
     * Find the data workspace for a diff workspace and corresponding data layer
     * name.
     * 
     * @param diffWorkspace
     * @param dataLayerName
     * @return
     */
    public static String getDataWorkspace(String diffWorkspace, String dataLayerName) {
        JDBCConnectionJNDI jdbcMeta = null;

        try {
            jdbcMeta = new JDBCConnectionJNDI("jdbc/georepoMetaResource");

            String sql1 = "SELECT dataworkspace FROM difftables WHERE diffworkspace=? and datalayername=?;";
            System.out.println(sql1);

            PreparedStatement ps1 = jdbcMeta.prepareStatement(sql1);
            ps1.setString(1, diffWorkspace);
            ps1.setString(2, dataLayerName);

            ResultSet rs1 = ps1.executeQuery();
            String dataWorkspace = null;

            // Check there is only one entry!
            int rsCount = 0;
            while (rs1.next()) {
                dataWorkspace = rs1.getString("dataworkspace");
                rsCount++;
            }

            if (rsCount > 1) {
                throw new InternalError("Data workspace cannot be determined uniquely!"
                        + " Multiple combinations of diffworkspace and datalayername exist" + " in table difftables.");
            }

            System.out.println("Data workspace for diff workspace" + diffWorkspace + " is " + dataWorkspace + ".");
            return dataWorkspace;

        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (jdbcMeta != null) {
                jdbcMeta.closeAll();
                jdbcMeta = null;
            }
        }
        return null;
    }

    public static String getDiffLayername(String dataWorkspace, String diffWorkspace, String dataLayerName) {
        JDBCConnectionJNDI jdbcMeta = null;

        try {
            jdbcMeta = new JDBCConnectionJNDI("jdbc/georepoMetaResource");

            String sql1 = "SELECT difflayername FROM difftables WHERE diffworkspace=? AND datalayername=?;";
            System.out.println(sql1);

            PreparedStatement ps1 = jdbcMeta.prepareStatement(sql1);
            ps1.setString(1, diffWorkspace);
            ps1.setString(2, dataLayerName);

            ResultSet rs1 = ps1.executeQuery();
            String diffLayername = null;

            int rsCount = 0;
            while (rs1.next()) {
                diffLayername = rs1.getString("difflayername");
                rsCount++;
            }
            if (rsCount > 1) {
                throw new InternalError("Diff layer cannot be determined uniquely!"
                        + " Multiple combinations of diffworkspace and datalayername exist in difftables.");
            }

            System.out.println("difflayername: " + diffLayername);

            return diffLayername;

        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (jdbcMeta != null) {
                jdbcMeta.closeAll();
                jdbcMeta = null;
            }
        }
        return null;
    }

    public static String getDataLayername(String diffWorkspace, String diffLayerName) {
        JDBCConnectionJNDI jdbcMeta = null;

        try {
            jdbcMeta = new JDBCConnectionJNDI("jdbc/georepoMetaResource");

            String sql1 = "SELECT datalayername FROM difftables WHERE diffworkspace=? AND difflayername=?;";
            System.out.println(sql1);

            PreparedStatement ps1 = jdbcMeta.prepareStatement(sql1);
            ps1.setString(1, diffWorkspace);
            ps1.setString(2, diffLayerName);

            ResultSet rs1 = ps1.executeQuery();
            String dataLayername = null;

            int rsCount = 0;
            while (rs1.next()) {
                dataLayername = rs1.getString("datalayername");
                rsCount++;
            }
            if (rsCount > 1) {
                throw new InternalError("Data layer cannot be determined uniquely!"
                        + " Multiple combinations of diffworkspace and difflayername exist in table difftables.");
            }

            System.out.println("dataLayername: " + dataLayername);

            return dataLayername;

        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (jdbcMeta != null) {
                jdbcMeta.closeAll();
                jdbcMeta = null;
            }
        }
        return null;
    }

    /**
     * Adds a diff layer to the difftables table.
     *
     * @return
     */
    public static boolean addDiffLayer(String diffLayername, String diffWorkspace, String dataLayerName,
            String dataWorkspace) {
        JDBCConnectionJNDI jdbcMeta = null;

        try {
            jdbcMeta = new JDBCConnectionJNDI("jdbc/georepoMetaResource");

            String sql = "INSERT INTO public.difftables (difflayername, diffworkspace, datalayername, dataworkspace)"
                    + " VALUES (?,?,?,?);";

            System.out.println(sql);

            PreparedStatement ps1 = jdbcMeta.prepareStatement(sql);
            ps1.setString(1, diffLayername);
            ps1.setString(2, diffWorkspace);
            ps1.setString(3, dataLayerName);
            ps1.setString(4, dataWorkspace);

            ps1.executeUpdate();

            return true;

        } catch (SQLException e) {
            e.printStackTrace();
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (jdbcMeta != null) {
                jdbcMeta.closeAll();
                jdbcMeta = null;
            }
        }
        return false;
    }

    /**
     * Creates a diff table from the structure of a data table. And adds a grpstatus
     * column.
     *
     * @param dataTable
     */
    public static void createDiffTable(String dataTable, String diffTable) throws SQLException, IOException {
        JDBCConnectionJNDI jdbcData = null;

        QueryCheck.checkTableName(dataTable);
        QueryCheck.checkTableName(diffTable);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql = "CREATE TABLE public." + diffTable + ""
                    + " (LIKE " + dataTable
                    + " INCLUDING DEFAULTS INCLUDING CONSTRAINTS INCLUDING INDEXES);"
                    + " ALTER TABLE " + diffTable
                    + " ADD COLUMN " + GeorepoConstants.GRPSTATUSCOLUMN + " character varying(1) NOT NULL;"
                    + " ALTER TABLE " + diffTable + " ALTER COLUMN " + GeorepoConstants.GFIDCOLUMN + " DROP NOT NULL;";

            System.out.println(sql);
            jdbcData.execute(sql);

        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    public static void createDiffAuthTable(String diffAuthTableName) throws SQLException, IOException {
        JDBCConnectionJNDI jdbcData = null;

        QueryCheck.checkTableName(diffAuthTableName);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql = "CREATE TABLE public." + diffAuthTableName
                    + "("
                    + "id SERIAL PRIMARY KEY,"
                    + "grpfid integer UNIQUE NOT NULL,"
                    + "insertedby integer NOT NULL,"
                    + "insertedat timestamp without time zone DEFAULT timezone('utc'::text, now()) NOT NULL"
                    + ");";
            System.out.println(sql);

            jdbcData.execute(sql);

        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    /**
     * Checks existence of a given diff table name. DO NOT use this on a
     * diffTableName that comes from user input!
     *
     * @param diffTableName
     * @return
     * @throws SQLException
     */
    public static boolean checkDiffTableExists(String diffTableName) throws SQLException, IOException {

        QueryCheck.checkTableName(diffTableName);

        JDBCConnectionJNDI jdbcData = null;

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            // Only use, if diffTableName does not come from user input
            // to prevent sql injection!!
            // PreparedStatements do not work on table name
            String sql = "SELECT 1 FROM " + diffTableName + ";";
            System.out.println(sql);

            jdbcData.executeQuery(sql);

            return true;
        } catch (Exception e) {
            e.printStackTrace();
            if (e.getMessage().contains("does not exist")) {
                return false;
            }
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }
}
