package de.narimo.georepo.server.api;

import java.io.IOException;
import java.io.InputStream;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.Consumes;
import javax.ws.rs.ForbiddenException;
import javax.ws.rs.GET;
import javax.ws.rs.NotFoundException;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.SecurityContext;
import javax.ws.rs.ext.Provider;

import de.narimo.commons.dto.RegistrationLanguage;
import de.narimo.commons.dto.User;
import de.narimo.geocore.ws.repository.UserRepository;
import de.narimo.georepo.server.api.app.AppManifestConfiguration;
import de.narimo.georepo.server.appconfig.AppConfiguration;
import de.narimo.georepo.server.appconfig.AppKey;
import de.narimo.georepo.server.appconfig.AppLayersConfiguration;
import de.narimo.georepo.server.appconfig.AppSettingsService;
import de.narimo.georepo.server.appconfig.ExtendedAppKey;
import de.narimo.georepo.server.noauth.RegistrationCheck;
import de.narimo.georepo.server.notification.Notifier;
import de.narimo.georepo.server.providers.InvoiceGenerator;
import de.narimo.georepo.server.repository.AppSettingsRepository;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;

@Provider
@Path("/apps")
@Tag(name = "AppSettings")
public class AppController {

    AppSettingsService settingsService = new AppSettingsService();

    @GET
    @Path("/")
    @Produces({ MediaType.APPLICATION_JSON })
    @Operation(summary = "Get appkeys activation information")
    public Response getAppKeys(
            @Context SecurityContext sec,
            @QueryParam("appkey") String appkey) {

        String username = sec.getUserPrincipal().getName();
        User user = UserRepository.getUser(username).get();

        try {
            if (appkey != null) {
                // get activation info for a specific appkey independent of requesting user
                ExtendedAppKey foundAppKey = settingsService.getPublicAppKeyInformation(appkey, user);
                return Response.ok().entity(foundAppKey).build();
            } else {
                // get all appkeys and their activation info for the requesting user
                List<AppKey> appkeys = settingsService.getMyAppKeys(user);
                return Response.ok().entity(appkeys).build();
            }
        } catch (NotFoundException e) {
            e.printStackTrace();
            return Response.status(Status.NOT_FOUND).build();
        } catch (Exception e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        }
    }

    @GET
    @Path("/{appkey}/details")
    @Produces({ MediaType.APPLICATION_JSON })
    @Operation(summary = "Get appdetails")
    public Response getAppDetails(
            @PathParam("appkey") String appkey) {

        try {
            AppDetailsConfiguration appdetails = settingsService.getAppDetailsConfiguration(appkey);
            return Response.ok().entity(appdetails.getAppDetails()).build();
        } catch (NotFoundException e) {
            e.printStackTrace();
            return Response.status(Status.NOT_FOUND).build();
        } catch (Exception e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        }
    }

    @GET
    @Path("/config")
    @Produces({ MediaType.APPLICATION_JSON })
    @Operation(summary = "Get an app configuration")
    public Response getAppConfiguration(
            @QueryParam("appkey") String appkey) {

        // We don't need the user since every authenticated user must be able to
        // retrieve an app's config even if it is not his, correct?
        try {
            AppConfiguration appConfiguration = settingsService.getAppConfiguration(appkey);
            return Response.ok().entity(appConfiguration).build();
        } catch (Exception e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        }
    }

    @GET
    @Path("/layers")
    @Produces({ MediaType.APPLICATION_JSON })
    @Operation(summary = "Get the app layer configuration")
    public Response getAppLayersConfiguration(
            @QueryParam("appkey") String appkey) {

        // We don't need the user since every authenticated user must be able to
        // retrieve an app's config even if it is not his, correct?
        try {
            AppLayersConfiguration appLayersConfiguration = settingsService.getAppLayersConfiguration(appkey);
            return Response.ok().entity(appLayersConfiguration).build();
        } catch (Exception e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        }
    }

    @GET
    @Path("/baselayers")
    @Produces({ MediaType.APPLICATION_JSON })
    @Operation(summary = "Get the app base layer configuration")
    public Response getAppBaseLayersConfiguration(
            @QueryParam("appkey") String appkey) {

        // We don't need the user since every authenticated user must be able to
        // retrieve an app's config even if it is not his, correct?
        try {
            AppBaseLayersConfiguration appbaseLayersConfiguration = settingsService
                    .getAppBaseLayersConfiguration(appkey);
            return Response.ok().entity(appbaseLayersConfiguration).build();
        } catch (NotFoundException e) {
            e.printStackTrace();
            return Response.status(Status.NOT_FOUND).build();
        } catch (Exception e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        }
    }

    @GET
    @Path("/overview")
    @Produces({ MediaType.APPLICATION_JSON })
    @Operation(summary = "Get all activated apps by template app")
    public Response getAppsOverview(
            @QueryParam("appkey") String templateKey) {

        try {
            List<AppKey> appkeys = settingsService.getAppKeysByTemplateKey(templateKey);
            return Response.ok().entity(appkeys).build();
        } catch (NotFoundException e) {
            e.printStackTrace();
            return Response.status(Status.NOT_FOUND).build();
        } catch (Exception e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        }
    }

    @POST
    @Path("/")
    @Consumes({ MediaType.APPLICATION_JSON })
    @Produces({ MediaType.APPLICATION_JSON })
    @Operation(summary = "Create a new appkey")
    public Response createAppKey(
            @Context ServletContext ctx,
            @Context SecurityContext sec,
            AppKey appkey) {

        String username = sec.getUserPrincipal().getName();
        User user = UserRepository.getUser(username).get();

        try {
            String key = settingsService.createAppKey(user, appkey);
            if (key == null) {
                throw new InternalError("Error creating new appkey");
            }

            AppDetailsConfiguration appDetails = new AppDetailsConfiguration();
            appDetails.setAppKey(key);
            appDetails.setAppDetails(new ArrayList<>());
            settingsService.saveAppDetailsConfiguration(appDetails);

            settingsService.createDefaultAppIcons(ctx, key);

            AppKey createdAppKey = new AppKey();
            createdAppKey.setAppKey(key);

            return Response.ok().entity(createdAppKey).build();

        } catch (IllegalStateException ise) {
            ise.printStackTrace();
            return Response.status(Status.CONFLICT).entity("App already contains icons.")
                    .type(MediaType.TEXT_PLAIN).build();
        } catch (NotFoundException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).entity("Template key is invalid.")
                    .type(MediaType.TEXT_PLAIN).build();
        } catch (SQLException | IOException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }

    @POST
    @Path("/activate")
    @Operation(summary = "Activate an appkey")
    public Response activateAppKey(
            @Context SecurityContext sec,
            @Context ServletContext ctx,
            @Context HttpServletRequest request,
            @QueryParam("appkey") String appkey) {

        String username = sec.getUserPrincipal().getName();
        User user = UserRepository.getUser(username).get();

        try {
            if (appkey == null) {
                throw new IllegalArgumentException();
            }

            if (!settingsService.isAppOwner(appkey, user)) {
                throw new ForbiddenException();
            }

            if (AppSettingsRepository.isAppkeyActivated(appkey, user)) {
                throw new IllegalStateException("The provided appkey is already activated.");
            }

            // Send the invoice, CC to the app provider and narimo
            String appAlias = null;
            String appOwnerEmail = null;
            RegistrationLanguage appOwnerLanguage = null;

            AppKey appKey = AppSettingsRepository.getAppKey(appkey);
            appKey.setAppKey(appkey);
            if (appkey != null) {
                appAlias = appKey.getAlias();
                User appOwner = UserRepository.getUsersById(Arrays.asList(appKey.getCreatedBy())).get(0);
                if (appOwner == null) {
                    // what to do, if the app owner has been deactivated or is not found?
                    // is this a relevant use case?
                    // right now, we just don't send a notification to the app owner
                } else {
                    appOwnerEmail = appOwner.getEmail();
                    appOwnerLanguage = appOwner.getLanguage();
                }
            }

            RegistrationCheck.doEmailCheck(appOwnerEmail);

            int providerUserId = AppSettingsRepository.getAppProvider(appKey.getAppKey(), user);
            User appProviderUser = UserRepository.getUsersById(Arrays.asList(providerUserId)).get(0);

            RegistrationCheck.doEmailCheck(appProviderUser.getEmail());

            Notifier notifier = Notifier.getInstance(ctx, appKey);

            InvoiceGenerator invoicer = settingsService.createInvoiceAndActivateAppkey(ctx, appkey, user);

            // send invoice to the customer
            notifier.sendGeorepoInvoice(appOwnerEmail, appAlias, invoicer.getAppProvider().getAlias(), appOwnerLanguage,
                    invoicer.getInvoiceName(),
                    invoicer.getSubstitutionMap().get("customername"), invoicer.getInvoicePath());

            // send invoice CC to the provider
            notifier.sendGeorepoInvoice(appProviderUser.getEmail(), appAlias, invoicer.getAppProvider().getAlias(),
                    appProviderUser.getLanguage(),
                    invoicer.getInvoiceName(), invoicer.getSubstitutionMap().get("customername"),
                    invoicer.getInvoicePath());

            // send invoice CC to narimo
            notifier.sendGeorepoInvoice("info@narimo.de", appAlias, invoicer.getAppProvider().getAlias(), null,
                    invoicer.getInvoiceName(),
                    invoicer.getSubstitutionMap().get("customername"), invoicer.getInvoicePath());

            // get activation info for a specific appkey independent of requesting user
            ExtendedAppKey foundAppKey = settingsService.getPublicAppKeyInformation(appkey, user);
            return Response.ok().entity(foundAppKey).build();

        } catch (IllegalArgumentException e) {
            e.printStackTrace();
            // deactivate the app again if we had already activated it
            AppSettingsRepository.deactivateAppkey(appkey, user);
            return Response.status(Status.BAD_REQUEST).entity("Appkey is invalid.")
                    .type(MediaType.TEXT_PLAIN).build();
        } catch (ForbiddenException e) {
            e.printStackTrace();
            // deactivate the app again if we had already activated it
            AppSettingsRepository.deactivateAppkey(appkey, user);

            return Response.status(Status.FORBIDDEN)
                    .entity("You don't have permissions to activate this appkey or this appkey does not exist.")
                    .type(MediaType.TEXT_PLAIN).build();
        } catch (IllegalStateException e) {
            e.printStackTrace();
            // do not deactivate the app in this error case!
            return Response.status(Status.CONFLICT).entity("The provided appkey is already activated.")
                    .type(MediaType.TEXT_PLAIN).build();
        } catch (RuntimeException e) {
            e.printStackTrace();
            // deactivate the app again if we had already activated it
            AppSettingsRepository.deactivateAppkey(appkey, user);

            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }

    @PUT
    @Path("/config")
    @Consumes({ MediaType.APPLICATION_JSON })
    @Produces({ MediaType.APPLICATION_JSON })
    @Operation(summary = "Save app configuration")
    public Response saveAppConfiguration(
            @Context SecurityContext sec,
            AppConfiguration appConfiguration) {

        try {
            String username = sec.getUserPrincipal().getName();
            User user = UserRepository.getUser(username).get();

            if (appConfiguration.getAppKey() == null) {
                throw new NotFoundException("Missing appkey in request.");
            }
            if (!settingsService.isAppOwner(appConfiguration.getAppKey(), user)) {
                throw new ForbiddenException("You don't have permissions to save these app settings.");
            }

            settingsService.saveAppConfiguration(appConfiguration);
            return Response.ok().build();
        } catch (NotFoundException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        } catch (SQLException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }

    @PUT
    @Path("/details")
    @Consumes({ MediaType.APPLICATION_JSON })
    @Produces({ MediaType.APPLICATION_JSON })
    @Operation(summary = "Save app description")
    public Response saveAppDetails(
            @Context SecurityContext sec,
            AppDetailsConfiguration appDetails) {

        try {
            String username = sec.getUserPrincipal().getName();
            User user = UserRepository.getUser(username).get();

            if (!settingsService.isAppOwner(appDetails.getAppKey(), user)) {
                throw new ForbiddenException("You don't have permissions to save these app settings.");
            }

            settingsService.saveAppDetailsConfiguration(appDetails);
            return Response.ok().build();
        } catch (NotFoundException | IllegalArgumentException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        } catch (SQLException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }

    @PUT
    @Path("/layers")
    @Consumes({ MediaType.APPLICATION_JSON })
    @Produces({ MediaType.APPLICATION_JSON })
    @Operation(summary = "Save app layer configuration")
    public Response saveAppLayersConfiguration(
            @Context SecurityContext sec,
            AppLayersConfiguration appLayersConfiguration) {

        try {
            String username = sec.getUserPrincipal().getName();
            User user = UserRepository.getUser(username).get();

            if (appLayersConfiguration == null || appLayersConfiguration.getAppKey() == null) {
                throw new NotFoundException("Missing appkey in request.");
            }
            if (!settingsService.isAppOwner(appLayersConfiguration.getAppKey(), user)) {
                throw new ForbiddenException("You don't have permissions to save these app settings.");
            }

            settingsService.saveAppLayersConfiguration(appLayersConfiguration);
            return Response.ok().build();
        } catch (NotFoundException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        } catch (SQLException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }

    @PUT
    @Path("/baselayers")
    @Consumes({ MediaType.APPLICATION_JSON })
    @Produces({ MediaType.APPLICATION_JSON })
    @Operation(summary = "Save app base layer configuration")
    public Response saveAppBaseLayersConfiguration(
            @Context SecurityContext sec,
            AppBaseLayersConfiguration appBaseLayersConfiguration) {

        try {
            String username = sec.getUserPrincipal().getName();
            User user = UserRepository.getUser(username).get();

            if (appBaseLayersConfiguration == null || appBaseLayersConfiguration.getAppKey() == null) {
                throw new NotFoundException("Missing appkey or payload in request.");
            }
            if (!settingsService.isAppOwner(appBaseLayersConfiguration.getAppKey(), user)) {
                throw new ForbiddenException("You don't have permissions to save these app settings.");
            }

            settingsService.saveAppBaseLayersConfiguration(appBaseLayersConfiguration);
            return Response.ok().build();
        } catch (NotFoundException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        } catch (SQLException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }

    /**
     * Add an app icon. This will create the different required icon formats.
     * 
     * @param sec
     * @param ctx
     * @param workspace
     * @return
     */
    @PUT
    @Path("/icons/{iconname}")
//    @Consumes({ "image/png", "image/jpg", "image/jpeg", "image/gif" })
    @Consumes({ "image/png" })
    @Operation(summary = "Save app icons")
    public Response addAppIcons(
            @Context SecurityContext sec,
            @Context ServletContext ctx,
            @PathParam("iconname") String iconName,
            @QueryParam("appkey") String appkey,
            InputStream uploadedInputStream) {

        try {

            String username = sec.getUserPrincipal().getName();
            User user = UserRepository.getUser(username).get();

            if (!settingsService.isAppOwner(appkey, user)) {
                throw new ForbiddenException("You don't have permissions to save these app settings.");
            }

            AppSettingsService appSettingsService = new AppSettingsService();
            appSettingsService.addAppIcons(ctx, uploadedInputStream, appkey, iconName);

            return Response.ok(Status.ACCEPTED).build();
        } catch (NotFoundException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        } finally {
            if (uploadedInputStream != null) {
                try {
                    uploadedInputStream.close();
                } catch (IOException e) {
                    // ignore
                }
            }
        }

    }

    @PUT
    @Path("/{appkey}/manifest")
    @Consumes({ MediaType.APPLICATION_JSON })
    @Operation(summary = "Update the manifest of an app")
    public Response addAppManifest(
            @Context SecurityContext sec,
            @PathParam("appkey") String appKey,
            AppManifestConfiguration appManifestConfiguration) {

        try {
            String username = sec.getUserPrincipal().getName();
            User user = UserRepository.getUser(username).get();

            if (appManifestConfiguration == null || appManifestConfiguration.getAppKey() == null) {
                throw new NotFoundException("Missing appkey or payload in request.");
            }
            if (!settingsService.isAppOwner(appManifestConfiguration.getAppKey(), user)) {
                throw new ForbiddenException("You don't have permissions to save these app settings.");
            }

            settingsService.saveAppManifest(appManifestConfiguration);
            return Response.ok().build();
        } catch (Exception e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        }
    }
}
