package de.narimo.georepo.server.api;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.NoSuchFileException;
import java.sql.SQLException;

import javax.servlet.ServletContext;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.ForbiddenException;
import javax.ws.rs.GET;
import javax.ws.rs.NotFoundException;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.SecurityContext;
import javax.ws.rs.ext.Provider;

import de.narimo.commons.dto.User;
import de.narimo.geocore.ws.repository.UserRepository;
import de.narimo.georepo.server.api.documents.DocumentType;
import de.narimo.georepo.server.api.documents.ImageAndDocumentService;
import de.narimo.georepo.server.repository.DatasetRepository;
import de.narimo.georepo.server.repository.DifftableRepository;
import de.narimo.georepo.server.tools.AdminTools;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;

@Provider
@Path("/workspaces/{diffWorkspace}/layers/{layername}/features/{gfid}/images/changes")
@Tag(name = "Images")
public class ImageModificationController {

    ImageAndDocumentService imageAndDocumentService = new ImageAndDocumentService();

    @GET
    @Path("/{filename}")
    @Produces({ "image/png", "image/jpg", "image/jpeg", "image/gif" })
    @Operation(summary = "Get a suggested image for a feature")
    public Response getUnreviewedImage(
            @Context SecurityContext sec,
            @Context ServletContext ctx,
            @PathParam("diffWorkspace") String diffWorkspace,
            @PathParam("layername") String dataLayerName,
            @PathParam("gfid") int gfid,
            @PathParam("filename") String fileName) {

        try {

            String workspace = DifftableRepository.getDataWorkspace(diffWorkspace, dataLayerName);
            if (workspace == null) {
                throw new InternalError("No data workspace defined.");
            }

            int layerId = DatasetRepository.getLayerId(workspace, dataLayerName);
            byte[] fileData = imageAndDocumentService.getDiffFeatureImageOrDocument(ctx, workspace, layerId, gfid,
                    fileName, DocumentType.IMAGE);

            return Response.ok(new ByteArrayInputStream(fileData)).build();

        } catch (NoSuchFileException | NotFoundException e) {
            return Response.status(Status.NOT_FOUND).build();
        } catch (SQLException | InternalError e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        } catch (IOException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).entity("Could not retrieve image. ").build();
        }
    }

    @POST
    @Path("")
    @Consumes({ "image/png", "image/jpg", "image/jpeg", "image/gif" })
    @Operation(summary = "Suggest adding a new image to a feature of a layer")
    public Response postLayerImage(
            @Context SecurityContext sec,
            @Context ServletContext ctx,
            @PathParam("diffWorkspace") String diffWorkspace,
            @PathParam("layername") String dataLayerName,
            @PathParam("gfid") int gfid,

            @QueryParam("filename") String filename,
            @QueryParam("description") String description,
            InputStream uploadedInputStream

    ) throws IOException, SQLException {

        try {

            User user = UserRepository.getUser(sec.getUserPrincipal().getName()).get();

            String workspace = DifftableRepository.getDataWorkspace(diffWorkspace, dataLayerName);
            if (workspace == null) {
                throw new InternalError("No data workspace defined.");
            }

            // If user has no write permission on that workspace, he would already have been
            // thrown out by authorization layer
            int layerId = DatasetRepository.getLayerId(workspace, dataLayerName);
            int diffImageId = imageAndDocumentService.addImageOrDocument(ctx, user, workspace, uploadedInputStream,
                    filename, layerId, gfid, DocumentType.IMAGE, description);

        } catch (SQLException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        } catch (IOException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).entity("Could not upload image. ").build();
        } finally {
            if (uploadedInputStream != null) {
                uploadedInputStream.close();
            }
        }

        return Response.status(201).build();
    }

    @DELETE
    @Path("")
    @Produces(MediaType.APPLICATION_JSON)
    @Operation(summary = "Suggest deleting an image from a feature of a layer")
    public static Response deleteLayerImage(
            @Context SecurityContext sec,
            @Context ServletContext ctx,
            @PathParam("diffWorkspace") String diffWorkspace,
            @PathParam("layername") String dataLayerName,
            @PathParam("gfid") String gfid) {

        return Response.status(Status.NOT_IMPLEMENTED).build();
    }

    @PUT
    @Path("/{filename}/accept")
    @Produces(MediaType.APPLICATION_JSON)
    @Operation(summary = "Accept a suggested image for a feature")
    public Response acceptImage(
            @Context SecurityContext sec,
            @Context ServletContext ctx,
            @PathParam("diffWorkspace") String diffWorkspace,
            @PathParam("layername") String dataLayerName,
            @PathParam("gfid") int gfid,
            @PathParam("filename") String diffDocumentFilename) {

        try {
            User user = UserRepository.getUser(sec.getUserPrincipal().getName()).get();

            String workspace = DifftableRepository.getDataWorkspace(diffWorkspace, dataLayerName);
            if (workspace == null) {
                throw new InternalError("No data workspace defined.");
            }

            AdminTools.checkAdminPermission(user.getId(), workspace);

            int layerId = DatasetRepository.getLayerId(workspace, dataLayerName);
            imageAndDocumentService.acceptImageOrDocument(ctx, layerId, gfid, diffDocumentFilename, workspace,
                    DocumentType.IMAGE);
            return Response.ok().build();

        } catch (NotFoundException e) {
            e.printStackTrace();
            return Response.status(Status.NOT_FOUND).build();
        } catch (IOException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        } catch (ForbiddenException e) {
            e.printStackTrace();
            return Response.status(Status.FORBIDDEN).build();
        } catch (InternalError | RuntimeException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }

    @PUT
    @Path("/{filename}/decline")
    @Produces(MediaType.APPLICATION_JSON)
    @Operation(summary = "Decline a suggested image for a feature")
    public Response declineImage(
            @Context SecurityContext sec,
            @Context ServletContext ctx,
            @PathParam("diffWorkspace") String diffWorkspace,
            @PathParam("layername") String dataLayerName,
            @PathParam("gfid") int gfid,
            @PathParam("filename") String diffDocumentFilename) {

        try {
            User user = UserRepository.getUser(sec.getUserPrincipal().getName()).get();

            String workspace = DifftableRepository.getDataWorkspace(diffWorkspace, dataLayerName);
            if (workspace == null) {
                throw new InternalError("No data workspace defined.");
            }

            AdminTools.checkAdminPermission(user.getId(), workspace);

            int layerId = DatasetRepository.getLayerId(workspace, dataLayerName);
            imageAndDocumentService.declineImageOrDocument(ctx, layerId, gfid, diffDocumentFilename, workspace,
                    DocumentType.IMAGE);
            return Response.ok().build();

        } catch (NotFoundException e) {
            e.printStackTrace();
            return Response.status(Status.NOT_FOUND).build();
        } catch (IOException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        } catch (ForbiddenException e) {
            e.printStackTrace();
            return Response.status(Status.FORBIDDEN).build();
        } catch (InternalError | RuntimeException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }
}
