package de.narimo.georepo.server.api;

import java.io.IOException;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import javax.servlet.ServletContext;
import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.SecurityContext;
import javax.ws.rs.ext.Provider;

import de.narimo.commons.dto.User;
import de.narimo.geocore.ws.repository.UserRepository;
import de.narimo.georepo.server.api.observations.Observation;
import de.narimo.georepo.server.api.observations.ObservationsService;
import de.narimo.georepo.server.repository.DatasetRepository;
import de.narimo.georepo.server.tools.AdminTools;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;

@Provider
@Path("/workspaces/{workspace}/layers/{layername}/features/{gfid}/observations")
@Tag(name = "Observations")
public class ObservationsController {

    ObservationsService observationsService = new ObservationsService();

    @GET
    @Path("/")
    @Produces({ MediaType.APPLICATION_JSON })
    @Operation(summary = "Retrieve observations")
    public Response getObservations(
            @Context SecurityContext sec,
            @PathParam("workspace") String workspace,
            @PathParam("layername") String dataLayerName,
            @PathParam("gfid") Integer gfid,
            @QueryParam("fromDate") String fromDate,
            @QueryParam("toDate") String toDate) {

        try {

            if (fromDate != null && fromDate.length() < 10) {
                throw new IllegalArgumentException(
                        "fromDate format must have format yyyy-mm-dd or yyyy-mm-dd hh:mm:ss");
            }
            if (toDate != null && toDate.length() < 10) {
                throw new IllegalArgumentException(
                        "toDate format must have format yyyy-mm-dd or yyyy-mm-dd hh:mm:ss");
            }

            int layerId = DatasetRepository.getLayerId(workspace, dataLayerName);

            List<Observation> observations = new ArrayList<>();
            if (fromDate != null && toDate != null) {
                observations = observationsService.getObservationsByTimeRange(layerId, gfid, fromDate,
                        toDate);
            } else {
                observations = observationsService.getObservationsByYears(layerId, gfid);
            }
            return Response.ok().entity(observations).build();

        } catch (SQLException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }

    @POST
    @Path("/")
    @Consumes({ MediaType.APPLICATION_JSON })
    @Operation(summary = "Add a new observation")
    public Response addObervation(
            @Context SecurityContext sec,
            @Context ServletContext ctx,
            @PathParam("workspace") String workspace,
            @PathParam("layername") String dataLayerName,
            @PathParam("gfid") Integer gfid,
            Observation observation) throws IOException, SQLException {

        try {

            User user = UserRepository.getUser(sec.getUserPrincipal().getName()).get();
            AdminTools.checkAdminPermission(user.getId(), workspace);

            int layerId = DatasetRepository.getLayerId(workspace, dataLayerName);
            observationsService.addObservation(layerId, user, observation);

        } catch (SQLException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }

        return Response.status(201).build();
    }
}
